<?php

namespace App\Http\Middleware;

use App\Enums\AccountStatus;
use App\Models\UserAccount;
use Closure;
use Illuminate\Http\Request;
use Log;
use Symfony\Component\HttpFoundation\Response;
use Tymon\JWTAuth\Exceptions\JWTException;
use Tymon\JWTAuth\Exceptions\TokenExpiredException;
use Tymon\JWTAuth\Exceptions\TokenInvalidException;
use Tymon\JWTAuth\Facades\JWTAuth;

class AuthorizeMiddleware
{
  /**
   * Handle an incoming request.
   *
   * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
   */

  protected array $allowedRoleTypes;

  public function __construct(array $allowedRoleTypes = [])
  {
    $this->allowedRoleTypes = $allowedRoleTypes;
  }
  public function handle(Request $request, Closure $next, ...$allowedRoleTypes): Response
  {
    try {
      // Get the token and authenticated user
      $decodedAuthToken = JWTAuth::parseToken()->authenticate();


      // Check if the user exists and is active
      $user = UserAccount::where('id', $decodedAuthToken->id)
        ->where('status', AccountStatus::APPROVED->value)
        ->first();
      if (!$user) {
        return response()->json(['message' => 'Invalid credentials. Please check your email and password.'], Response::HTTP_UNAUTHORIZED);
      }

      // Check if the role is allowed
      if (!in_array($user->role_id, $allowedRoleTypes)) {
        return response()->json(['message' => 'Access Denied: You do not have the required permissions to perform this action.'], Response::HTTP_FORBIDDEN);
      }

      // Attach user details to the request
      $request->user = $decodedAuthToken;
    } catch (TokenExpiredException $e) {
      return response()->json(['message' => 'Session expired. Please log in again to continue.'], Response::HTTP_UNAUTHORIZED);
    } catch (TokenInvalidException $e) {
      return response()->json(['message' => 'Invalid session token. Please log in to continue.'], Response::HTTP_UNAUTHORIZED);
    } catch (JWTException $e) {
      return response()->json(['message' => 'Authorization token not provided. Please log in to access this resource.'], Response::HTTP_UNAUTHORIZED);
    }

    return $next($request);
  }
}
