<?php

namespace App\Http\Controllers;

use App\Enums\Role;
use App\Http\Controllers\Controller;
use App\Models\DoctorBooking;
use App\Models\UserAccount;
use App\Models\VideoConsultation;
use App\Services\BMeetService;
use App\Traits\ApiResponse;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Tymon\JWTAuth\Facades\JWTAuth;

class VideoConsultationController extends Controller
{
  protected $bmeetService;
  use ApiResponse;

  public function __construct(BMeetService $bmeetService)
  {
    $this->bmeetService = $bmeetService;
  }

  /**
   * Create a new meeting for a booking
   */
  public function createMeeting($bookingId)
  {
    try {
      $userToken = JWTAuth::parseToken()->getPayload();
      $userAccountId = $userToken->get('userAccountId');
      $roleId = $userToken->get('roleId');
      // Check if booking exists and user is authorized
      $booking = DoctorBooking::findOrFail($bookingId);

      // Only admins and doctors should create meetings
      if ($roleId != Role::ADMIN && $userAccountId != $booking->doctor_id) {
        return $this->error(401, __('message.unauthorized'));
      }

      // Check if meeting already exists
      if ($booking->videoConsultation) {
        return response()->json([
          'message' => 'Meeting already exists',
          'data' => $booking->videoConsultation
        ]);
      }

      // Create a meeting
      $meetingData = $this->bmeetService->createMeeting($booking);

      // Save the consultation details
      $consultation = VideoConsultation::create([
        'booking_id' => $booking->id,
        'meeting_id' => $meetingData['mid'],
        'm_id' => $meetingData['m_id'],
        'scheduled_start' => Carbon::parse($booking->booking_date),
        'scheduled_end' => Carbon::parse($booking->booking_date)->addMinutes(30),
      ]);

      return $this->success(
        __(key: 'message.success'),
        $consultation
      );
    } catch (\Exception $e) {
      Log::error('Create meeting error: ' . $e->getMessage());
      return response()->json(['message' => 'Failed to create meeting', 'error' => $e->getMessage()], 500);
    }
  }

  /**
   * Get meeting details for a doctor
   */
  public function getDoctorMeetingDetails($bookingId)
  {
    try {
      $userToken = JWTAuth::parseToken()->getPayload();
      $userAccountId = $userToken->get('userAccountId');
      $user = UserAccount::with(['loginData'])
        ->where('id', $userAccountId)
        ->firstOrFail();

      $booking = DoctorBooking::with('videoConsultation')->findOrFail($bookingId);

      // Verify doctor is authorized
      if ($userAccountId != $booking->doctor_id) {
        return $this->error(401, __('message.unauthorized'));
      }

      // Check if meeting exists
      if (!$booking->videoConsultation) {
        return response()->json(['message' => 'No video consultation scheduled'], 404);
      }

      // Get doctor's name
      $doctorName = $user->loginData->login_name;

      // Get the meeting details with moderator privileges
      $meetingDetails = $this->bmeetService->getModeratorMeetingDetails(
        $booking->videoConsultation->meeting_id,
        $doctorName
      );

      return $this->success(
        __(key: 'message.success'),
        [
          'meetingDetails' => $meetingDetails,
          'participantName' => $doctorName,
          'role' => 'moderator',
          'mid' => $booking->videoConsultation->meeting_id,
        ]
      );
    } catch (\Exception $e) {
      Log::error('Get doctor meeting details error: ' . $e->getMessage());
      return response()->json(['message' => 'Failed to get meeting details', 'error' => $e->getMessage()], 500);
    }
  }

  /**
   * Get meeting details for a patient
   */
  public function getPatientMeetingDetails($bookingId)
  {
    try {
      $userToken = JWTAuth::parseToken()->getPayload();
      $userAccountId = $userToken->get('userAccountId');
      $user = UserAccount::with(['loginData'])
        ->where('id', $userAccountId)
        ->firstOrFail();


      $booking = DoctorBooking::with('videoConsultation')->findOrFail($bookingId);

      // Verify patient is authorized
      if ($userAccountId != $booking->patient_id) {
        return $this->error(401, __('message.unauthorized'));
      }

      // Check if meeting exists
      if (!$booking->videoConsultation) {
        return response()->json(['message' => 'No video consultation scheduled'], 404);
      }

      // Get patient's name
      $patientName = $user->loginData->login_name;

      // Get the meeting details (without moderator privileges)
      $meetingDetails = $this->bmeetService->getMeetingDetails(
        $booking->videoConsultation->meeting_id,
        $patientName
      );

      return $this->success(
        __(key: 'message.success'),
        [
          'meetingDetails' => $meetingDetails,
          'participantName' => $patientName,
          'role' => 'participant',
          'mid' => $booking->videoConsultation->meeting_id,
        ]
      );
    } catch (\Exception $e) {
      Log::error('Get patient meeting details error: ' . $e->getMessage());
      return response()->json(['message' => 'Failed to get meeting details', 'error' => $e->getMessage()], 500);
    }
  }

  /**
   * Check join status for participants waiting approval
   */
  public function checkJoinStatus($participantId)
  {
    try {
      $status = $this->bmeetService->getJoinStatus($participantId);
      return response()->json($status);
    } catch (\Exception $e) {
      Log::error('Check join status error: ' . $e->getMessage());
      return response()->json(['message' => 'Failed to check join status', 'error' => $e->getMessage()], 500);
    }
  }

  /**
   * Get meeting credentials for either doctor or patient
   */
  public function getMeetingCredentials($bookingId)
  {
    try {
      $userToken = JWTAuth::parseToken()->getPayload();
      $userAccountId = $userToken->get('userAccountId');
      $user = UserAccount::with(['loginData'])
        ->where('id', $userAccountId)
        ->firstOrFail();

      $booking = DoctorBooking::with('videoConsultation')->findOrFail($bookingId);

      $isDoctor = $userAccountId == $booking->doctor_id;
      $isPatient = $userAccountId == $booking->patient_id;

      if (!$isDoctor && !$isPatient) {
        return $this->error(401, __('message.unauthorized'));
      }

      // Check if meeting exists
      if (!$booking->videoConsultation) {
        return response()->json(['message' => 'No video consultation scheduled'], 404);
      }

      // Get meeting info based on role
      if ($isDoctor) {
        $meetingDetails = $this->bmeetService->getModeratorMeetingDetails(
          $booking->videoConsultation->meeting_id,
          $user->loginData->login_name
        );
        $role = 'moderator';
      } else {
        $meetingDetails = $this->bmeetService->getMeetingDetails(
          $booking->videoConsultation->meeting_id,
          $user->loginData->login_name
        );
        $role = 'participant';
      }

      return $this->success(
        __(key: 'message.success'),
        [
          'meetingDetails' => $meetingDetails,
          'participantName' => $user->login_name,
          'role' => $role,
          'conference_url' => $booking->videoConsultation->conference_url,
          'meeting_id' => $booking->videoConsultation->meeting_id,
          'm_id' => $booking->videoConsultation->m_id
        ]
      );
    } catch (\Exception $e) {
      Log::error('Get meeting credentials error: ' . $e->getMessage());
      return response()->json(['message' => 'Failed to get meeting credentials', 'error' => $e->getMessage()], 500);
    }
  }
}
