<?php

namespace App\Http\Controllers;

use App\Exceptions\AppException;
use App\Http\Controllers\Controller;
use App\Models\SpecializationService;
use App\Traits\ApiResponse;
use Dotenv\Exception\ValidationException;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Validator;

class SpecializationServiceController extends Controller
{
  use ApiResponse;
  public function index(Request $request, $specializationId)
  {
    try {
      $validator = Validator::make($request->all(), [
        'page' => 'integer|min:1',
        'size' => 'integer|min:1|max:100',
        'search' => 'string|nullable'
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();
      $page = (int) ($validated['page'] ?? 1);
      $size = (int) ($validated['size'] ?? 10);
      $search = $validated['search'] ?? null;
      $offset = ($page - 1) * $size;

      $query = SpecializationService::where('specialization_id', $specializationId);

      if (!empty($search)) {
        $query->where('name', 'like', '%' . $search . '%');
      }

      $total = $query->count();

      $services = $query->orderBy('created_at', 'desc')
        ->offset($offset)
        ->limit($size)
        ->get();

      return $this->success(__('message.success'), [
        'services' => $services,
        'pagination' => [
          'total' => $total,
          'current_page' => $page,
          'per_page' => $size,
          'last_page' => ceil($total / $size)
        ]
      ], 200);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  public function show($specializationId, $id)
  {
    try {
      $service = SpecializationService::where('specialization_id', $specializationId)->where('id', $id)->firstOrFail();
      return $this->success(__('message.success'), $service, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.spcl_service_not_found'));
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  public function store(Request $request, $specializationId)
  {
    try {
      $validator = Validator::make($request->all(), [
        'name' => 'required|string|max:255',
        'description' => 'nullable|string',
        'name_ar' => 'nullable|string|max:255',
        'description_ar' => 'nullable|string',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();
      $validated['specialization_id'] = $specializationId;
      $service = SpecializationService::create($validated);

      return $this->success(__('message.spcl_service_created'), $service, 201);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  public function update(Request $request, $specializationId, $id)
  {
    try {
      $service = SpecializationService::where('specialization_id', $specializationId)->where('id', $id)->firstOrFail();

      $validator = Validator::make($request->all(), [
        'name' => 'required|string|max:255',
        'description' => 'nullable|string',
        'name_ar' => 'nullable|string|max:255',
        'description_ar' => 'nullable|string'
      ]);


      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();
      $service->update($validated);

      return $this->success(__('message.spcl_service_updated'), $service, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.spcl_service_not_found'), []);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  public function destroy($specializationId, $id)
  {
    try {
      $service = SpecializationService::where('specialization_id', $specializationId)
        ->where('id', $id)
        ->firstOrFail();

      $service->delete();
      return $this->success(__('message.spcl_service_deleted'), null, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.spcl_service_not_found'));
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  public function publicIndex($id)
  {


    try {
      $services = SpecializationService::select('id', 'name', 'name_ar')->where('specialization_id', $id)->get();
      return $this->success(__('message.success'), $services, 200);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
}
