<?php

namespace App\Http\Controllers;

use App\Exceptions\AppException;
use App\Http\Controllers\Controller;
use App\Models\DoctorSlot;
use App\Traits\ApiResponse;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Validator;

class SlotController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    use ApiResponse;
    public function index(Request $request, string $doctorId)
    {
      try {
        $validator = Validator::make($request->query(), [
          'page' => 'integer|min:1',
          'size' => 'integer|min:1|max:100',
        ]);
  
        if ($validator->fails()) {
          throw new AppException(__('message.validation_error'), 400, $validator->errors());
        }
  
        $validated = $validator->validated();
        $page = (int) ($validated['page'] ?? 1);
        $size = (int) ($validated['size'] ?? 10);
        $offset = ($page - 1) * $size;
  
        $query = DoctorSlot::where('doctor_id', $doctorId);
  
   
  
        $total = $query->count();
  
        $slots = $query->orderBy('created_at', 'desc')->with([
          'workingDay:id,working_day',
          'workingHour:id,start_time,end_time'
      ])
          ->offset($offset)
          ->limit($size)
          ->get();
  
        return $this->success(__('message.success'), [
          'slots' => $slots,
          'pagination' => [
            'total' => $total,
            'current_page' => $page,
            'per_page' => $size,
            'last_page' => ceil($total / $size)
          ]
        ], 200);
      } catch (AppException $e) {
        return $e->render($request);
      } catch (Exception $e) {
        return $this->error(500, __('message.server_error'), [$e->getMessage()]);
      }
    }

   
    public function show(string $doctorId,string $id,)
    {
      try {
        $slot = DoctorSlot::where('doctor_id', $doctorId)
    ->where('id', $id)
    ->with([
        'workingDay:id,working_day',
        'workingHour:id,start_time,end_time'
    ])
    ->findOrFail($id);
        return $this->success(__('message.success'), $slot, 200);
      } catch (ModelNotFoundException $e) {
        return $this->error(404, __('message.not_found'), []);
      } catch (Exception $e) {
        return $this->error(500, __('message.server_error'), [$e->getMessage()]);
      }
    }

  
    public function destroy(string $doctorId,string $id)
    {
      try {
        $service = DoctorSlot::where('doctor_id', $doctorId)
          ->where('id', $id)
          ->firstOrFail();
  
        $service->delete();
        return $this->success(__('message.success'), null, 200);
      } catch (ModelNotFoundException $e) {
        return $this->error(404, __('message.not_found'));
      } catch (Exception $e) {
        return $this->error(500, __('message.server_error'), [$e->getMessage()]);
      }
    }
}
