<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Plan;
use Illuminate\Http\Request;

class PlanController extends Controller
{
  public function index()
  {
    $plans = Plan::all();
    return response()->json($plans, 200);
  }

  /**
   * Store a new plan.
   */
  public function store(Request $request)
  {
    $validated = $request->validate([
      'name' => 'required|string|max:255',
      'description' => 'nullable|string',
      'price' => 'required|numeric',
      'currency' => 'required|string|max:3',
      'duration_months' => 'required|integer|min:1',
      'status' => 'required|exists:plan_status,id',
      'is_trial' => 'boolean',
    ]);

    $plan = Plan::create($validated);
    return response()->json(['message' => 'Plan created successfully', 'plan' => $plan], 201);
  }

  /**
   * Get a single plan by ID.
   */
  public function show($id)
  {
    $plan = Plan::find($id);
    if (!$plan) {
      return response()->json(['message' => 'Plan not found'], 404);
    }
    return response()->json($plan, 200);
  }

  /**
   * Update a plan.
   */
  public function update(Request $request, $id)
  {
    $plan = Plan::find($id);
    if (!$plan) {
      return response()->json(['message' => 'Plan not found'], 404);
    }

    $validated = $request->validate([
      'name' => 'sometimes|string|max:255',
      'description' => 'sometimes|string',
      'price' => 'sometimes|numeric',
      'currency' => 'sometimes|string|max:3',
      'duration_months' => 'sometimes|integer|min:1',
      'status' => 'sometimes|exists:plan_status,id',
      'is_trial' => 'boolean',
    ]);

    $plan->update($validated);
    return response()->json(['message' => 'Plan updated successfully', 'plan' => $plan], 200);
  }

  /**
   * Delete a plan.
   */
  public function destroy($id)
  {
    $plan = Plan::find($id);
    if (!$plan) {
      return response()->json(['message' => 'Plan not found'], 404);
    }
    $plan->delete();
    return response()->json(['message' => 'Plan deleted successfully'], 200);
  }

  /**
   * Change plan status.
   */
  public function action(Request $request, $id)
  {
    $plan = Plan::find($id);
    if (!$plan) {
      return response()->json(['message' => 'Plan not found'], 404);
    }

    $request->validate([
      'status' => 'required|exists:plan_status,id',
    ]);

    $plan->update(['status' => $request->status]);

    return response()->json(['message' => 'Plan status updated successfully', 'plan' => $plan], 200);
  }
}
