<?php

namespace App\Http\Controllers;

use App\Enums\Role;
use App\Exceptions\AppException;
use App\Http\Controllers\Controller;
use App\Models\UserAccount;
use App\Models\UserLoginData;
use App\Traits\ApiResponse;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Validator;

class PatientController
{
  use ApiResponse;
  public function index(Request $request)
  {
    try {
      $validator = Validator::make($request->all(), [
        'page' => 'integer|min:1',
        'size' => 'integer|min:1|max:100',
        'search' => 'string|nullable'
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();
      $page = (int) ($validated['page'] ?? 1);
      $size = (int) ($validated['size'] ?? 10);
      $search = $validated['search'] ?? null;
      $offset = ($page - 1) * $size;

      $patientsQuery = UserAccount::from('user_account as ua')
        ->select(
          'ua.id',
          'ua.avatar_url',
          'ua.first_name',
          'ua.last_name',
          'ua.country_code',
          'ua.mobile_number',
          'ua.emergency_country_code',
          'ua.emergency_mobile_number',
          'ua.national_id',
          'ua.national_id_exp_at',
          'ua.status',
          'ul.email',
          'ul.login_name',
          'ua.created_at',
          'ua.updated_at',
        )
        ->leftJoin('user_login_data as ul', 'ua.id', '=', 'ul.user_account_id')
        ->where('ua.role_id', Role::USER->value);

      if (!empty($search)) {
        $searchTerm = "%{$search}%";
        $patientsQuery->where(
          fn($query) =>
          $query->where('ua.first_name', 'LIKE', $searchTerm)
            ->orWhere('ua.last_name', 'LIKE', $searchTerm)
        );
      }

      $total = $patientsQuery->count();

      $patients = $patientsQuery
        ->orderBy('ua.created_at', 'desc')
        ->offset($offset)
        ->limit($size)
        ->get();

      return $this->success(__('message.success'), [
        'patients' => $patients,
        'pagination' => [
          'total' => $total,
          'current_page' => $page,
          'per_page' => $size,
          'last_page' => ceil($total / $size)
        ]
      ], 200);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  /**
   * Store a newly created resource in storage.
   */
  public function store(Request $request)
  {
    //
  }

  /**
   * Display the specified resource.
   */
  public function show(string $id)
  {
    try {
      $patient = UserAccount::from('user_account as ua')
        ->select(
          'ua.id',
          'ua.avatar_url',
          'ua.first_name',
          'ua.last_name',
          'ua.country_code',
          'ua.mobile_number',
          'ua.emergency_country_code',
          'ua.emergency_mobile_number',
          'ua.national_id',
          'ua.national_id_exp_at',
          'ua.status',
          'ul.email',
          'ul.login_name',
          'ua.created_at',
          'ua.updated_at',
        )
        ->leftJoin('user_login_data as ul', 'ua.id', '=', 'ul.user_account_id')
        ->where('ua.role_id', Role::USER->value)
        ->where('ua.id', $id)
        ->firstOrFail();


      return $this->success(__('message.success'), $patient, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'));
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  /**
   * Update the specified resource in storage.
   */
  public function profileUpdate(Request $request, string $id)
  {
    DB::beginTransaction();
    try {
      $patient = UserAccount::from('user_account as ua')
        ->select('ua.*', 'ul.email', 'ul.login_name')
        ->leftJoin('user_login_data as ul', 'ua.id', '=', 'ul.user_account_id')
        ->where('ua.role_id', Role::DOCTOR->value)
        ->where('ua.id', $id)
        ->firstOrFail();


      $validator = Validator::make($request->all(), [
        'avatar' => 'required|file|mimes:jpg,jpeg,png|max:20480',
        'username' => 'required|string|max:255',
        'firstName' => 'required|string|max:255',
        'lastName' => 'required|string|max:255',
        'nationalId' => 'required|string|unique:user_account,national_id,' . $id,
        'nationalIdExpiryAt' => 'required|date',
        'countryCode' => 'required|string',
        'mobile' => 'required|string|unique:user_account,mobile_number,' . $id,
        'emergencyCountryCode' => 'required|string',
        'emergencyMobile' => 'required|string',
        'email' => 'required|email|unique:user_login_data,email,' . $id . ',user_account_id',
        'dateOfBirth' => 'required|date',
        'gender' => 'required|string',
        'password' => 'required|string|min:6',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();

      $userAccountUpdate = [];
      $userLoginDataUpdate = [];

      if (isset($validated['avatar'])) {
        $avatar = $validated['avatar'];
        $fileName = time() . '-' . $avatar->getClientOriginalName();
        $avatar->storeAs('profile', $fileName, 'public');
        $path = '/storage/profile/' . $fileName;

        $userAccountUpdate['avatar_url'] = $path;
      }
      if (isset($validated['firstName'])) {
        $userAccountUpdate['first_name'] = $validated['firstName'];
      }
      if (isset($validated['lastName'])) {
        $userAccountUpdate['last_name'] = $validated['lastName'];
      }
      if (isset($validated['nationalId'])) {
        $userAccountUpdate['national_id'] = $validated['nationalId'];
      }
      if (isset($validated['nationalIdExpiryAt'])) {
        $userAccountUpdate['national_id_exp_at'] = $validated['nationalIdExpiryAt'];
      }
      if (isset($validated['countryCode'])) {
        $userAccountUpdate['country_code'] = $validated['countryCode'];
      }
      if (isset($validated['mobile'])) {
        $userAccountUpdate['mobile_number'] = $validated['mobile'];
      }
      if (isset($validated['emergencyCountryCode'])) {
        $userAccountUpdate['emergency_country_code'] = $validated['emergencyCountryCode'];
      }
      if (isset($validated['emergencyMobile'])) {
        $userAccountUpdate['emergency_mobile_number'] = $validated['emergencyMobile'];
      }
      if (isset($validated['dateOfBirth'])) {
        $userAccountUpdate['date_of_birth'] = $validated['dateOfBirth'];
      }
      if (isset($validated['gender'])) {
        $userAccountUpdate['gender'] = $validated['gender'];
      }

      if (isset($validated['email'])) {
        $userLoginDataUpdate['email'] = $validated['email'];
      }
      if (isset($validated['password'])) {
        $userLoginDataUpdate['password_hash'] = Hash::make($validated['password']);
      }
      if (isset($validated['username'])) {
        $userLoginDataUpdate['login_name'] = strtolower($validated['username']);
      }

      UserAccount::where('id', $id)->update($userAccountUpdate);
      UserLoginData::where('user_account_id', $id)->update($userLoginDataUpdate);

      $uPatient = UserAccount::find($id);

      DB::commit();
      return $this->success(__('message.Patient details updated successfully!'), $uPatient, 200);
    } catch (ModelNotFoundException $e) {
      DB::rollBack();
      return $this->error(404, __('message.not_found'));
    } catch (AppException $e) {
      DB::rollBack();
      return $e->render($request);
    } catch (Exception $e) {
      DB::rollBack();
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  /**
   * Remove the specified resource from storage.
   */
  public function destroy(string $id)
  {
    try {
      $patient = UserAccount::where('role_id', Role::USER->value)
        ->where('id', $id)
        ->firstOrFail();


      $patient->delete();

      return $this->success(__('message.patient_deleted_successfully'), [], 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'));
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  public function action(Request $request, $id)
  {
    try {
      $patient = UserAccount::where('role_id', Role::USER->value)
        ->where('id', $id)
        ->firstOrFail();

      $validator = Validator::make($request->all(), [
        'status' => 'required|exists:account_status,id',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }
      $validated = $validator->validated();

      $patient->update(['status' => $validated['status']]);

      return $this->success(__('message.success'), $patient, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'));
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function update(Request $request, string $id)
  {
    DB::beginTransaction();
    try {
      UserAccount::from('user_account as ua')
        ->select('ua.*', 'ul.email', 'ul.login_name')
        ->leftJoin('user_login_data as ul', 'ua.id', '=', 'ul.user_account_id')
        ->where('ua.role_id', Role::USER->value)
        ->where('ua.id', $id)
        ->firstOrFail();


      $validator = Validator::make($request->all(), [
        'username' => 'required|string|max:255|unique:user_login_data,login_name,' . $id . ',user_account_id',
        'firstName' => 'required|string|max:255',
        'lastName' => 'required|string|max:255',
        'nationalId' => 'required|string|unique:user_account,national_id,' . $id,
        'nationalIdExpiryAt' => 'required|date',
        'countryCode' => 'required|string',
        'mobile' => 'required|string|unique:user_account,mobile_number,' . $id,
        'emergencyCountryCode' => 'required|string',
        'emergencyMobile' => 'required|string',
        'email' => 'required|email|unique:user_login_data,email,' . $id . ',user_account_id',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();

      $userAccountUpdate = [];
      $userLoginDataUpdate = [];

      if (isset($validated['firstName'])) {
        $userAccountUpdate['first_name'] = $validated['firstName'];
      }
      if (isset($validated['lastName'])) {
        $userAccountUpdate['last_name'] = $validated['lastName'];
      }
      if (isset($validated['nationalId'])) {
        $userAccountUpdate['national_id'] = $validated['nationalId'];
      }
      if (isset($validated['nationalIdExpiryAt'])) {
        $userAccountUpdate['national_id_exp_at'] = $validated['nationalIdExpiryAt'];
      }
      if (isset($validated['countryCode'])) {
        $userAccountUpdate['country_code'] = $validated['countryCode'];
      }
      if (isset($validated['mobile'])) {
        $userAccountUpdate['mobile_number'] = $validated['mobile'];
      }
      if (isset($validated['emergencyCountryCode'])) {
        $userAccountUpdate['emergency_country_code'] = $validated['emergencyCountryCode'];
      }
      if (isset($validated['emergencyMobile'])) {
        $userAccountUpdate['emergency_mobile_number'] = $validated['emergencyMobile'];
      }

      if (isset($validated['email'])) {
        $userLoginDataUpdate['email'] = $validated['email'];
      }
      if (isset($validated['username'])) {
        $userLoginDataUpdate['login_name'] = strtolower($validated['username']);
      }

      UserAccount::where('id', $id)->update($userAccountUpdate);
      UserLoginData::where('user_account_id', $id)->update($userLoginDataUpdate);

      $updatedPatient = UserAccount::from('user_account as ua')
        ->select('ua.*', 'ul.email', 'ul.login_name')
        ->leftJoin('user_login_data as ul', 'ua.id', '=', 'ul.user_account_id')
        ->where('ua.id', $id)
        ->first();

      DB::commit();
      return $this->success(__('message.patient_updated'), $updatedPatient, 200);
    } catch (ModelNotFoundException $e) {
      DB::rollBack();
      return $this->error(404, __('message.not_found'));
    } catch (AppException $e) {
      DB::rollBack();
      return $e->render($request);
    } catch (Exception $e) {
      DB::rollBack();
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
}
