<?php

namespace App\Http\Controllers;

use App\Enums\DoctorAvailabilityStatus;
use App\Enums\Role;
use App\Exceptions\AppException;
use App\Http\Controllers\Controller;
use App\Models\DoctorWorkingHour;
use App\Models\UserAccount;
use App\Traits\ApiResponse;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Log;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class DoctorWorkingHourController extends Controller
{

  use ApiResponse;
  public function index(Request $request)
  {
    try {

      $validator = Validator::make($request->query(), [
        'page' => 'integer|min:1',
        'size' => 'integer|min:1|max:100',
        'doctorId' => 'nullable|exists:doctor,user_account_id',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();
      $page = (int) ($validated['page'] ?? 1);
      $size = (int) ($validated['size'] ?? 10);
      $doctorId = $validated['doctorId'] ?? null;
      $offset = ($page - 1) * $size;

      $workingHourQuery = DoctorWorkingHour::query();

      if (!empty($doctorId)) {
        $workingHourQuery->where('doctor_id', $doctorId);
      };


      $total = $workingHourQuery->count();

      $workingHours = $workingHourQuery
        ->orderBy('created_at', 'desc')
        ->offset($offset)
        ->limit($size)
        ->get();
      return $this->success(__(key: 'message.success'), [
        'working_hours' => $workingHours,
        'pagination' => [
          'total' => $total,
          'current_page' => $page,
          'per_page' => $size,
          'last_page' => ceil($total / $size)
        ]
      ], 200);
    } catch (AppException $e) {
      Log::info('Request received AppException', [$e]);
      return $e->render($request);
    } catch (Exception $e) {
      Log::info('Request received Exception', [$e]);
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function show($id)
  {
    try {
      $department = DoctorWorkingHour::findOrFail($id);
      return $this->success(__('message.success'), $department, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'), []);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  /**
   * Store a newly created resource in storage.
   */
  public function store(Request $request)
  {
    try {
      $validator = Validator::make($request->all(), [
        'doctorId'  => 'required|exists:doctor,user_account_id',
        'startTime' => 'required|date_format:H:i:s',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();

      // Get doctor details
      $doctor = UserAccount::from('user_account as ua')
        ->select(
          'ua.id',
          'ua.first_name',
          'ua.last_name',
          'ua.mobile_number',
          'ua.avatar_url',
          'ua.created_at',
          'd.avrg_consultation_time',
          'd.avrg_consultation_fee'
        )
        ->leftJoin('doctor as d', 'ua.id', '=', 'd.user_account_id')
        ->where('ua.role_id', Role::DOCTOR->value)
        ->where('ua.id', $validated['doctorId'])
        ->firstOrFail();

      if (!$doctor->avrg_consultation_time) {
        throw new AppException(__('message.consultation_time_not_set'), 400);
      }

      // Convert start_time to Carbon instance
      $startTime = Carbon::createFromFormat('H:i:s', $validated['startTime']);

      // Calculate end_time by adding consultation time (in minutes)
      $endTime = $startTime->copy()->addMinutes($doctor->avrg_consultation_time);

      // Check for existing time slot conflicts
      $conflict = DoctorWorkingHour::where('doctor_id', $doctor->id)
        ->where(function ($query) use ($startTime, $endTime) {
          $query->whereBetween('start_time', [$startTime, $endTime])
            ->orWhereBetween('end_time', [$startTime, $endTime])
            ->orWhere(function ($q) use ($startTime, $endTime) {
              $q->where('start_time', '<=', $startTime)
                ->where('end_time', '>=', $endTime);
            });
        })
        ->exists();

      if ($conflict) {
        throw new AppException(__('message.time_slot_conflict'), 400);
      }

      // Save the working hour
      $availability = DoctorWorkingHour::create([
        'doctor_id'  => $doctor->id,
        'start_time' => $startTime,
        'end_time'   => $endTime,
        'status'     =>  DoctorAvailabilityStatus::AVAILABLE->value,
      ]);

      return $this->success(__('message.success'), $availability, 201);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'));
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  public function update(Request $request, string $id)
  {
    try {
      $workingHour = DoctorWorkingHour::findOrFail($id);

      $validator = Validator::make($request->all(), [
        'doctorId'  => 'required|exists:doctor,user_account_id',
        'startTime' => 'required|date_format:H:i:s',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();

      // Get doctor details
      $doctor = UserAccount::from('user_account as ua')
        ->select(
          'ua.id',
          'ua.first_name',
          'ua.last_name',
          'ua.mobile_number',
          'ua.avatar_url',
          'ua.created_at',
          'd.avrg_consultation_time',
          'd.avrg_consultation_fee'
        )
        ->leftJoin('doctor as d', 'ua.id', '=', 'd.user_account_id')
        ->where('ua.role_id', Role::DOCTOR->value)
        ->where('ua.id', $validated['doctorId'])
        ->firstOrFail();

      if (!$doctor->avrg_consultation_time) {
        throw new AppException(__('message.consultation_time_not_set'), 400);
      }

      // Convert start_time to Carbon instance
      $startTime = Carbon::createFromFormat('H:i:s', $validated['startTime']);

      // Calculate end_time by adding consultation time
      $endTime = $startTime->copy()->addMinutes($doctor->avrg_consultation_time);

      // Check for existing time slot conflicts (excluding the current record)
      $conflict = DoctorWorkingHour::where('doctor_id', $doctor->id)
        ->where('id', '!=', $workingHour->id) // Exclude the current record
        ->where(function ($query) use ($startTime, $endTime) {
          $query->whereBetween('start_time', [$startTime, $endTime])
            ->orWhereBetween('end_time', [$startTime, $endTime])
            ->orWhere(function ($q) use ($startTime, $endTime) {
              $q->where('start_time', '<=', $startTime)
                ->where('end_time', '>=', $endTime);
            });
        })
        ->exists();

      if ($conflict) {
        throw new AppException(__('message.time_slot_conflict'), 400);
      }

      // Update the working hour
      $workingHour->update([
        'doctor_id'  => $doctor->id,
        'start_time' => $startTime,
        'end_time'   => $endTime,
        'status'     => DoctorAvailabilityStatus::AVAILABLE->value,
      ]);

      return $this->success(__('message.working_hour_updated'), $workingHour, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'), []);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }



  /**
   * Remove the specified resource from storage.
   */
  public function destroy(string $id)
  {
    try {
      $department = DoctorWorkingHour::findOrFail($id);
      $department->delete();
      return $this->success(__('message.success'), null, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'));
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  // public function publicIndex()
  // {
  //   try {
  //     $departments = DoctorWorkingHour::select('id', 'doctor_id', 'start_time', 'end_time')->get();

  //     return $this->success(__('message.success'), $departments, 200);
  //   } catch (Exception $e) {
  //     return $this->error(500, __('message.server_error'), [$e->getMessage()]);
  //   }
  // }
}
