<?php

namespace App\Http\Controllers;

use App\Enums\DoctorAvailabilityStatus;
use App\Enums\DoctorSlotStatus;
use App\Enums\Role;
use App\Exceptions\AppException;
use App\Http\Controllers\Controller;
use App\Http\Resources\AvailableDayListResource;
use App\Models\DoctorSlot;
use App\Models\DoctorWorkingDay;
use App\Traits\ApiResponse;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Tymon\JWTAuth\Facades\JWTAuth;

class DoctorSlotController extends Controller
{
  use ApiResponse;
  /**
   * Display a listing of the resource.
   */
  public function index(Request $request, $doctorId = null)
  {
    try {
      $user = JWTAuth::parseToken()->getPayload();
      $role = $user->get('roleId');

      if ($role === Role::DOCTOR->value) {
        $doctorId = $user->get('userAccountId');
      }
      if (!$doctorId) {
        throw new AppException(__('message.doctor_id_required'), 400);
      }

      $validator = Validator::make($request->query(), [
        'startDate' => 'nullable|date',
        'endDate'   => 'nullable|date|after_or_equal:start_date',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }
      $validated = $validator->validated();
      $startDate = $validated['startDate'] ?? null;
      $endDate = $validated['endDate'] ?? null;
      $currentTime = Carbon::now();

      $appointmentDaysQuery = DoctorWorkingDay::where('doctor_id', $doctorId)
        ->when(
          $startDate && $endDate,
          fn($query) => $query->whereBetween('working_day', [$startDate, $endDate])
        )
        ->with([
          'slots' => fn($query) => $query
            ->when(
              $role == Role::USER->value,
              fn($query) => $query->where('status', DoctorSlotStatus::OPEN->value)
            )
        ])
        ->when(
          $role == Role::USER->value,
          fn($query) => $query->where('status', DoctorAvailabilityStatus::AVAILABLE->value)
            ->where('working_day', '>', $currentTime)
        )
        ->orderBy('working_day', 'asc');

      $appointmentDays = $appointmentDaysQuery->get()
        ->filter(fn($day) => $day->slots->isNotEmpty())
        ->values();


      return $this->success(__('message.success'), ['appointment_days' => $appointmentDays->isEmpty() ? [] : AvailableDayListResource::collection($appointmentDays)], 200);
    } catch (AppException $e) {

      return $e->render($request);
    } catch (Exception $e) {
      return response()->json(['error' => __('message.server_error'), 'details' => $e->getMessage()], 500);
    }
  }

  /**
   * Store a newly created resource in storage.
   */
  public function store(Request $request, $doctorId = null)
  {
    DB::beginTransaction();
    try {
      $user = JWTAuth::parseToken()->getPayload();

      if ($user->get('roleId') === Role::DOCTOR->value) {
        $doctorId = $user->get('userAccountId');
      }
      if (!$doctorId) {
        throw new AppException(__('message.doctor_id_required'), 400);
      }

      $validator = Validator::make($request->all(), [
        'startDate' => 'required|date',
        'endDate' => 'required|date|after_or_equal:startDate',
        'slots' => 'required|array',
        'slots.*.id' => 'required|exists:doctor_working_hour,id',
      ]);
      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();

      $startDate = Carbon::parse($validated['startDate']);
      $endDate = Carbon::parse($validated['endDate']);
      $createdWorkingDays = [];

      while ($startDate->lte($endDate)) {
        $currentDate = $startDate->format('Y-m-d');

        $workingDay = DoctorWorkingDay::where('doctor_id', $doctorId)
          ->where('working_day', $currentDate)
          ->first();

        if (!$workingDay) {
          $workingDay = DoctorWorkingDay::create([
            'doctor_id' => $doctorId,
            'working_day' => $currentDate,
            'status' => DoctorSlotStatus::OPEN->value
          ]);
        }
        $createdWorkingDays[] = [
          'id' => $workingDay->id,
          'doctor_id' => $doctorId,
          'working_day' => $workingDay->working_day,
          'status' => $workingDay->status,
        ];
        DoctorSlot::where('doctor_id', $doctorId)
          ->where('working_day_id', $workingDay->id)
          ->delete();

        foreach ($validated['slots'] as $slot) {
          DoctorSlot::create([
            'doctor_id' => $doctorId,
            'working_day_id' => $workingDay->id,
            'working_hour_id' => $slot['id'],
            'status' => DoctorSlotStatus::OPEN->value
          ]);
        }

        $startDate->addDay();
      }

      DB::commit();
      return $this->success(__('message.doctor_slot_created'), $createdWorkingDays, 201);
    } catch (AppException $e) {
      DB::rollBack();
      return $e->render($request);
    } catch (Exception $e) {
      DB::rollBack();
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  public function destroy(Request $request, $id)
  {
    try {
      $user = JWTAuth::parseToken()->getPayload();

      if ($user->get('roleId') === Role::DOCTOR->value) {
        $doctorId = $user->get('userAccountId');
      }
      $query = DoctorWorkingDay::where('id', $id);

      if (!empty($doctorId)) {
        $query->where('doctor_id', $doctorId);
      }

      $workingDay = $query->firstOrFail();

      $workingDay->delete();

      return $this->success(__('message.doctor_slot_deleted'), null, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'));
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
}
