<?php

namespace App\Http\Controllers;

use App\Enums\DoctorAvailabilityStatus;
use App\Enums\Role;
use App\Exceptions\AppException;
use App\Http\Controllers\Controller;
use App\Models\DoctorWorkingHour;
use App\Models\UserAccount;
use App\Traits\ApiResponse;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Tymon\JWTAuth\Facades\JWTAuth;

class DoctorScheduleController extends Controller
{
  use ApiResponse;
  /**
   * Display a listing of the resource.
   */
  public function index(Request $request, $doctorId = null)
  {
    try {
      $user = JWTAuth::parseToken()->getPayload();

      if ($user->get('roleId') === Role::DOCTOR->value) {
        $doctorId = $user->get('userAccountId');
      }
      if (!$doctorId) {
        throw new AppException(__('message.doctor_id_required'), 400);
      }

      $availability = DoctorWorkingHour::query()->where('doctor_id', $doctorId)
        ->where('status', DoctorAvailabilityStatus::AVAILABLE->value)
        ->orderBy('created_at', 'desc')
        ->get();
      return $this->success(__(key: 'message.success'), [
        'availability' => $availability
      ], 200);
    } catch (AppException $e) {
      Log::info('Request received AppException', [$e]);
      return $e->render($request);
    } catch (Exception $e) {
      Log::info('Request received Exception', [$e]);
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  /**
   * Store a newly created resource in storage.
   */
  public function store(Request $request, $doctorId = null)
  {
    try {

      $user = JWTAuth::parseToken()->getPayload();

      if ($user->get('roleId') === Role::DOCTOR->value) {
        $doctorId = $user->get('userAccountId');
      }
      if (!$doctorId) {
        throw new AppException(__('message.doctor_id_required'), 400);
      }

      $validator = Validator::make($request->all(), [
        'startTime' => 'required|date_format:H:i:s',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();

      // Get doctor details
      $doctor = UserAccount::from('user_account as ua')
        ->select(
          'ua.id',
          'ua.first_name',
          'ua.last_name',
          'ua.mobile_number',
          'ua.avatar_url',
          'ua.created_at',
          'd.avrg_consultation_time',
          'd.avrg_consultation_fee'
        )
        ->leftJoin('doctor as d', 'ua.id', '=', 'd.user_account_id')
        ->where('ua.role_id', Role::DOCTOR->value)
        ->where('ua.id', $doctorId)
        ->firstOrFail();

      if (!$doctor->avrg_consultation_time) {
        throw new AppException(__('message.consultation_time_not_set'), 400);
      }

      // Convert start_time to Carbon instance
      $startTime = Carbon::createFromFormat('H:i:s', $validated['startTime']);

      // Calculate end_time by adding consultation time (in minutes)
      $endTime = $startTime->copy()->addMinutes($doctor->avrg_consultation_time);

      // Check for existing time slot conflicts
      $conflict = DoctorWorkingHour::where('doctor_id', $doctor->id)
        ->where(function ($query) use ($startTime, $endTime) {
          $query->whereBetween('start_time', [$startTime, $endTime])
            ->orWhereBetween('end_time', [$startTime, $endTime])
            ->orWhere(function ($q) use ($startTime, $endTime) {
              $q->where('start_time', '<=', $startTime)
                ->where('end_time', '>=', $endTime);
            });
        })
        ->exists();

      if ($conflict) {
        throw new AppException(__('message.time_slot_conflict'), 400);
      }

      // Save the working hour
      $availability = DoctorWorkingHour::create([
        'doctor_id'  => $doctor->id,
        'start_time' => $startTime,
        'end_time'   => $endTime,
        'status'     =>  DoctorAvailabilityStatus::AVAILABLE->value,
      ]);

      return $this->success(__('message.doctor_availabilty_created'), $availability, 201);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'));
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  public function destroy(Request $request,  $id)
  {
    try {
      $user = JWTAuth::parseToken()->getPayload();

      if ($user->get('roleId') === Role::DOCTOR->value) {
        $doctorId = $user->get('userAccountId');
      }
      $query = DoctorWorkingHour::where('id', $id);

      if (!empty($doctorId)) {
        $query->where('doctor_id', $doctorId);
      }

      $availability = $query->firstOrFail();
      $availability->delete();
      return $this->success(__('message.doctor_availabilty_deleted'), null, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'));
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
}
