<?php

namespace App\Http\Controllers;

use App\Enums\AccountStatus;
use App\Enums\DoctorSlotStatus;
use App\Enums\Role;
use App\Exceptions\AppException;
use App\Http\Controllers\Controller;
use App\Http\Resources\DoctorResource;
use App\Mail\UserApprovedMail;
use App\Mail\UserDeletededMail;
use App\Mail\UserRejectedMail;
use App\Models\Doctor;
use App\Models\DoctorBooking;
use App\Models\DoctorCertificate;
use App\Models\DoctorIdsDocument;
use App\Models\DoctorReview;
use App\Models\DoctorSlot;
use App\Models\DoctorWorkingDay;
use App\Models\ReviewLike;
use App\Models\UserAccount;
use App\Models\UserLoginData;
use App\Traits\ApiResponse;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Log;
use Illuminate\Support\Facades\Validator;
use Tymon\JWTAuth\Facades\JWTAuth;

class DoctorController extends Controller
{
  use ApiResponse;
  public function index(Request $request)
  {
    try {
      $user = JWTAuth::parseToken()->getPayload();
      $role = $user->get('roleId');

      $validator = Validator::make($request->all(), [
        'page' => 'integer|min:1',
        'size' => 'integer|min:1|max:100',
        'search' => 'string|nullable',
        'departmentId' => 'integer|nullable',
        'specializationId' => 'integer|nullable',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();
      $page = (int) ($validated['page'] ?? 1);
      $size = (int) ($validated['size'] ?? 10);
      $search = $validated['search'] ?? null;
      $departmentId = $validated['departmentId'] ?? null;
      $specializationId = $validated['specializationId'] ?? null;
      $offset = ($page - 1) * $size;

      $doctorsQuery = UserAccount::from('user_account as ua')
        ->select(
          'ua.id',
          'ua.avatar_url',
          'ua.first_name',
          'ua.last_name',
          'ua.country_code',
          'ua.mobile_number',
          'ua.emergency_country_code',
          'ua.emergency_mobile_number',
          'ua.national_id',
          'ua.national_id_exp_at',
          'ua.status',
          'ul.email',
          'ul.login_name',
          'dr.license_no',
          'dr.department_id',
          'dr.designation_id',
          'dr.specialization_id',
          'dr.specialization_service_id',
          'dr.avrg_consultation_time',
          'dr.avrg_consultation_fee',
          'dr.qualification',
          'dr.medical_school',
          'dr.experience',
          'dr.avrg_consultation_fee',
          'dep.name as department_name',
          'dep.name_ar as department_name_ar',
          's.name as specialization_name',
          's.name_ar as specialization_name_ar',
          'des.name as designation_name',
          'des.name_ar as designation_name_ar',
          'ua.created_at',
          'ua.updated_at',
        )
        ->leftJoin('user_login_data as ul', 'ua.id', '=', 'ul.user_account_id')
        ->leftJoin('doctor as dr', 'ua.id', '=', 'dr.user_account_id')
        ->leftJoin('department as dep', 'dr.department_id', '=', 'dep.id')
        ->leftJoin('specialization as s', 'dr.specialization_id', '=', 's.id')
        ->leftJoin('designation as des', 'dr.designation_id', '=', 'des.id')
        ->where('ua.role_id', Role::DOCTOR->value);

      if (!empty($search)) {
        $searchTerm = "%{$search}%";
        $doctorsQuery->where(
          fn($query) =>
          $query->where('ua.first_name', 'LIKE', $searchTerm)
            ->orWhere('ua.last_name', 'LIKE', $searchTerm)
        );
      }

      if (!empty($departmentId)) {
        $doctorsQuery->where('dr.department_id', $departmentId);
      }

      if (!empty($specializationId)) {
        $doctorsQuery->where('dr.specialization_id', $specializationId);
      }
      if ($role == Role::USER->value) {
        $doctorsQuery->where('ua.status', AccountStatus::APPROVED->value);
      }

      $total = $doctorsQuery->count();

      $doctors = $doctorsQuery
        ->orderBy('ua.created_at', 'desc')
        ->offset($offset)
        ->limit($size)
        ->get();


      return $this->success(__('message.success'), [
        'doctors' => $doctors,
        'pagination' => [
          'total' => $total,
          'current_page' => $page,
          'per_page' => $size,
          'last_page' => ceil($total / $size)
        ]
      ], 200);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function store(Request $request)
  {
    //
    Log::info('DoctorController@store method hit');
    return response()->json([
      'message' => 'Strore method was hit successfully!'
    ]);
  }


  public function show(string $id)
  {
    try {
      $doctor = UserAccount::from('user_account as ua')
        ->select(
          'ua.id',
          'ua.avatar_url',
          'ua.first_name',
          'ua.last_name',
          'ua.country_code',
          'ua.mobile_number',
          'ua.emergency_country_code',
          'ua.emergency_mobile_number',
          'ua.national_id',
          'ua.national_id_exp_at',
          'ua.status',
          'ul.email',
          'ul.login_name',
          'dr.license_no',
          'dr.department_id',
          'dr.designation_id',
          'dr.specialization_service_id',
          'dr.avrg_consultation_time',
          'dr.avrg_consultation_fee',
          'dr.qualification',
          'dr.medical_school',
          'dr.experience',
          'dr.avrg_consultation_fee',
          'dep.name as department_name',
          'ua.created_at',
          'ua.updated_at',
        )
        ->leftJoin('user_login_data as ul', 'ua.id', '=', 'ul.user_account_id')
        ->leftJoin('doctor as dr', 'ua.id', '=', 'dr.user_account_id')
        ->leftJoin('department as dep', 'dr.department_id', '=', 'dep.id')
        ->where('ua.role_id', Role::DOCTOR->value)
        ->where('ua.id', $id)
        ->firstOrFail();

      return $this->success(__('message.success'), $doctor, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'));
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function showDetails(string $id)
  {
    try {
      $user = JWTAuth::parseToken()->getPayload();
      $role = $user->get('roleId');

      $query = UserAccount::with([
        'doctor',
        'department',
        'specialization',
        'nextAvailableSlot.workingDay',
        'nextAvailableSlot.workingHour'
      ])
        ->where('role_id', Role::DOCTOR->value)
        ->where('id', $id);

      if ($role == Role::USER->value) {
        $query->where('status', AccountStatus::APPROVED->value);
      }

      $doctor = $query->firstOrFail()->makeHidden([
        'created_at',
        'updated_at',
        'role_id',
        'national_id',
        'national_id_exp_at'
      ]);

      return $this->success(__('message.success'), new DoctorResource($doctor), 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'));
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  /**
   * Update the specified resource in storage.
   */
  public function updateProfile(Request $request, string $id)
  {
    try {
      $doctor = UserAccount::from('user_account as ua')
        ->select('ua.*')
        ->leftJoin('doctor as d', 'ua.id', '=', 'd.user_account_id')
        ->addSelect('d.*')
        ->where('ua.role_id', Role::DOCTOR->value)
        ->where('ua.id', $id)
        ->firstOrFail();

      $validator = Validator::make($request->all(), [
        'avatar' => 'required|file|mimes:jpg,jpeg,png|max:20480',
        'firstName' => 'required|string|max:255',
        'lastName' => 'required|string|max:255',
        'nationalId' => 'required|string|unique:user_account,national_id',
        'nationalIdExpiryAt' => 'required|date',
        'countryCode' => 'required|string',
        'mobile' => 'required|string|unique:user_account,mobile_number',
        'emergencyCountryCode' => 'required|string',
        'emergencyMobile' => 'required|string',
        'email' => 'required|email|unique:user_login_data,email',
        'dateOfBirth' => 'required|date',
        'gender' => 'required|string',
        'password' => 'required|string|min:6',
        'licenseNo' => 'required|string|max:255|regex:/^[A-Za-z0-9\-]+$/',
        'specializationId' => 'required|integer',
        'experience' => 'required|numeric|min:1|max:99',
        'qualification' => 'required|string|max:255',
        'medicalSchool' => 'required|string|max:255',
        'departmentId' => 'required|integer',
        'designationId' => 'required|integer',
        'bio' => 'required|string|max:1055',
        'areasOfExpertise' => 'required|string|max:1055',
        'medicalMalpractice' => 'required|file|mimes:jpg,jpeg,png,pdf,doc,docx|max:20480',
        'certifications' => 'required|array',
        'certifications.*' => 'file|mimes:pdf,jpg,jpeg,png|max:20480',
        'identificationDocuments' => 'required|array',
        'identificationDocuments.*' => 'file|mimes:pdf,jpg,jpeg,png|max:20480',
        'specializationServiceId' => 'required|integer',
        'avrgConsultationTime' => 'required|integer|min:15|max:300',
        'avrgConsultationFee' => 'required|numeric|min:0.01|max:10000'
      ]);
      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();

      // Update user details
      $userAccountUpdate = [];
      if (isset($validated['avatar'])) {
        $avatar = $validated['avatar'];
        $fileName = time() . '-' . $avatar->getClientOriginalName();
        $avatar->storeAs('profile', $fileName, 'public');
        $path = '/storage/profile/' . $fileName;

        $userAccountUpdate['avatar_url'] = $path;
      }
      if (isset($validated['firstName'])) {
        $userAccountUpdate['first_name'] = $validated['licenseNo'];
      }

      if (isset($validated['lastName'])) {
        $userAccountUpdate['last_name'] = $validated['specializationId'];
      }

      if (isset($validated['nationalId'])) {
        $userAccountUpdate['national_id'] = $validated['experience'];
      }
      if (isset($validated['nationalIdExpiryAt'])) {
        $userAccountUpdate['national_id_exp_at'] = $validated['nationalIdExpiryAt'];
      }
      if (isset($validated['countryCode'])) {
        $userAccountUpdate['country_code'] = $validated['countryCode'];
      }
      if (isset($validated['mobile'])) {
        $userAccountUpdate['mobile_number'] = $validated['mobile'];
      }
      if (isset($validated['emergencyCountryCode'])) {
        $userAccountUpdate['emergency_country_code'] = $validated['emergencyCountryCode'];
      }
      if (isset($validated['emergencyMobile'])) {
        $userAccountUpdate['emergency_mobile_number'] = $validated['emergencyMobile'];
      }
      if (isset($validated['email'])) {
        $userAccountUpdate['email'] = $validated['email'];
      }
      if (isset($validated['dateOfBirth'])) {
        $userAccountUpdate['date_of_birth'] = $validated['dateOfBirth'];
      }
      if (isset($validated['gender'])) {
        $userAccountUpdate['gender'] = $validated['gender'];
      }
      if (isset($validated['password'])) {
        $userAccountUpdate['password_hash'] = Hash::make($validated['password']);
      }
      UserAccount::where('id', $id)->update($userAccountUpdate);
      $doctorUpdate = [];

      if (isset($validated['licenseNo'])) {
        $doctorUpdate['license_no'] = $validated['licenseNo'];
      }

      if (isset($validated['specializationId'])) {
        $doctorUpdate['specialization_id'] = $validated['specializationId'];
      }

      if (isset($validated['experience'])) {
        $doctorUpdate['experience'] = $validated['experience'];
      }

      if (isset($validated['qualification'])) {
        $doctorUpdate['qualification'] = $validated['qualification'];
      }

      if (isset($validated['medicalSchool'])) {
        $doctorUpdate['medical_school'] = $validated['medicalSchool'];
      }

      if (isset($validated['departmentId'])) {
        $doctorUpdate['department_id'] = $validated['departmentId'];
      }

      if (isset($validated['designationId'])) {
        $doctorUpdate['designation_id'] = $validated['designationId'];
      }

      if (isset($validated['bio'])) {
        $doctorUpdate['bio'] = $validated['bio'];
      }

      if (isset($validated['areasOfExpertise'])) {
        $doctorUpdate['areas_of_expertise'] = $validated['areasOfExpertise'];
      }

      if (isset($validated['specializationServiceId'])) {
        $doctorUpdate['specialization_service_id'] = $validated['specializationServiceId'];
      }

      if (isset($validated['avrgConsultationTime'])) {
        $doctorUpdate['avrg_consultation_time'] = $validated['avrgConsultationTime'];
      }

      if (isset($validated['avrgConsultationFee'])) {
        $doctorUpdate['avrg_consultation_fee'] = $validated['avrgConsultationFee'];
      }

      if (isset($validated['medicalMalpractice'])) {
        $malpractice = $validated['medicalMalpractice'];
        $malpracticeFileName = time() . '-' . $malpractice->getClientOriginalName();
        $malpractice->storeAs('doctor/medical_malpractice', $malpracticeFileName);
        $doctorUpdate['medical_malpractice'] = $malpracticeFileName;
      }

      Doctor::where('user_account_id', $id)->update($doctorUpdate);
      if ($request->hasFile('certifications')) {
        DoctorCertificate::where('doctor_id', $id)->delete();
        foreach ($request->file('certifications') as $file) {
          $path = $file->store('doctor/certifications');
          DoctorCertificate::create([
            'doctor_id' => $id,
            'original_name' => $file->getClientOriginalName(),
            'file_url' => $path,
            'mime_type' => $file->getMimeType(),
            'size' => $file->getSize(),
          ]);
        }
      }
      if ($request->hasFile('identificationDocuments')) {
        DoctorIdsDocument::where('doctor_id', $id)->delete();
        foreach ($request->file('identificationDocuments') as $file) {
          $path = $file->store('doctor/identifications');
          DoctorIdsDocument::create([
            'doctor_id' => $id,
            'original_name' => $file->getClientOriginalName(),
            'file_url' => $path,
            'mime_type' => $file->getMimeType(),
            'size' => $file->getSize(),
          ]);
        }
      }
      return $this->success(__('message.Doctor profile updated successfully!'), $doctor, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'));
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function update(Request $request, string $id)
  {
    DB::beginTransaction();
    try {
      $doctor = UserAccount::from('user_account as ua')
        ->select('ua.*', 'ul.email', 'ul.login_name')
        ->leftJoin('user_login_data as ul', 'ua.id', '=', 'ul.user_account_id')
        ->where('ua.role_id', Role::DOCTOR->value)
        ->where('ua.id', $id)
        ->firstOrFail();

      log::info('asd', ['asd' => $request->all()]);
      $validator = Validator::make($request->all(), [
        'username' => 'required|string|max:255|unique:user_login_data,login_name,' . $id . ',user_account_id',
        'email' => 'required|email|unique:user_login_data,email,' . $id . ',user_account_id',
        'firstName' => 'required|string|max:255',
        'lastName' => 'required|string|max:255',
        'nationalId' => 'required|string|unique:user_account,national_id,' . $id,
        'nationalIdExpiryAt' => 'required|date',
        'countryCode' => 'required|string',
        'mobile' => 'required|string|unique:user_account,mobile_number,' . $id,
        'emergencyCountryCode' => 'required|string',
        'emergencyMobile' => 'required|string',
        'licenseNo' => 'required|string|max:255|regex:/^[A-Za-z0-9\-]+$/',
        'specializationId' => 'required|integer',
        'departmentId' => 'required|integer',
        'designationId' => 'required|integer',
        'specializationServiceId' => 'required|integer',
        'avrgConsultationTime' => 'required|integer|min:15|max:300',
        'avrgConsultationFee' => 'required|numeric|min:0.01|max:10000'
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();

      $userAccountUpdate = [];
      $userLoginDataUpdate = [];
      $doctorUpdate = [];

      if (isset($validated['firstName'])) {
        $userAccountUpdate['first_name'] = $validated['firstName'];
      }
      if (isset($validated['lastName'])) {
        $userAccountUpdate['last_name'] = $validated['lastName'];
      }
      if (isset($validated['nationalId'])) {
        $userAccountUpdate['national_id'] = $validated['nationalId'];
      }
      if (isset($validated['nationalIdExpiryAt'])) {
        $userAccountUpdate['national_id_exp_at'] = $validated['nationalIdExpiryAt'];
      }
      if (isset($validated['countryCode'])) {
        $userAccountUpdate['country_code'] = $validated['countryCode'];
      }
      if (isset($validated['mobile'])) {
        $userAccountUpdate['mobile_number'] = $validated['mobile'];
      }
      if (isset($validated['emergencyCountryCode'])) {
        $userAccountUpdate['emergency_country_code'] = $validated['emergencyCountryCode'];
      }
      if (isset($validated['emergencyMobile'])) {
        $userAccountUpdate['emergency_mobile_number'] = $validated['emergencyMobile'];
      }

      if (isset($validated['email'])) {
        $userLoginDataUpdate['email'] = $validated['email'];
      }
      if (isset($validated['username'])) {
        $userLoginDataUpdate['login_name'] = strtolower($validated['username']);
      }
      if (isset($validated['licenseNo'])) {
        $doctorUpdate['license_no'] = $validated['licenseNo'];
      }
      if (isset($validated['specializationId'])) {
        $doctorUpdate['specialization_id'] = $validated['specializationId'];
      }
      if (isset($validated['departmentId'])) {
        $doctorUpdate['department_id'] = $validated['departmentId'];
      }

      if (isset($validated['designationId'])) {
        $doctorUpdate['designation_id'] = $validated['designationId'];
      }

      if (isset($validated['specializationServiceId'])) {
        $doctorUpdate['specialization_service_id'] = $validated['specializationServiceId'];
      }

      if (isset($validated['avrgConsultationTime'])) {
        $doctorUpdate['avrg_consultation_time'] = $validated['avrgConsultationTime'];
      }

      if (isset($validated['avrgConsultationFee'])) {
        $doctorUpdate['avrg_consultation_fee'] = $validated['avrgConsultationFee'];
      }

      if (!empty($userAccountUpdate)) {
        UserAccount::where('id', $id)->update($userAccountUpdate);
      }
      if (!empty($userLoginDataUpdate)) {
        UserLoginData::where('user_account_id', $id)->update($userLoginDataUpdate);
      }
      if (!empty($doctorUpdate)) {
        Doctor::where('user_account_id', $id)->update($doctorUpdate);
      }

      $updatedDoctor = UserAccount::from('user_account as ua')
        ->select(
          'ua.id',
          'ua.first_name',
          'ua.last_name',
          'ua.avatar_url',
          'ua.national_id',
          'ua.national_id_exp_at',
          'ua.country_code',
          'ua.mobile_number',
          'ua.emergency_country_code',
          'ua.emergency_mobile_number',
          'ul.email',
          'ul.login_name',
          'd.license_no',
          'd.specialization_id',
          'd.department_id',
          'd.designation_id',
          'd.specialization_service_id',
          'd.avrg_consultation_time',
          'd.avrg_consultation_fee',
          'ua.created_at',
          'ua.updated_at',
        )
        ->leftJoin('user_login_data as ul', 'ua.id', '=', 'ul.user_account_id')
        ->leftJoin('doctor as d', 'ua.id', '=', 'd.user_account_id')
        ->where('ua.id', $id)
        ->first();

      DB::commit();
      return $this->success(__('message.doctor_updated'), $updatedDoctor, 200);
    } catch (ModelNotFoundException $e) {
      DB::rollBack();
      return $this->error(404, __('message.not_found'));
    } catch (AppException $e) {
      DB::rollBack();
      return $e->render($request);
    } catch (Exception $e) {
      DB::rollBack();
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  /**
   * Remove the specified resource from storage.
   */
  public function destroy(string $id)
  {
    try {
      $doctor = UserAccount::where('role_id', Role::DOCTOR->value)
        ->where('id', $id)
        ->firstOrFail();


      $doctor->delete();

      return $this->success(__('message.doctor_deleted_successfully'), [], 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'));
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function action(Request $request, $id)
  {
    try {
      $doctor = UserAccount::where('role_id', Role::DOCTOR->value)
        ->where('id', $id)
        ->firstOrFail();


      $validator = Validator::make($request->all(), [
        'status' => 'required|exists:account_status,id',
        'note' => 'nullable|string|max:1000',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }
      $validated = $validator->validated();

      if ($doctor->status !== AccountStatus::PENDING->value && in_array($validated['status'], [AccountStatus::APPROVED->value, AccountStatus::REJECTED->value])) {
        throw new AppException(__('message.invalid_status_action'), 400);
      }

      if ($doctor->status == AccountStatus::APPROVED->value || $doctor->status == AccountStatus::REJECTED->value && $validated['status'] != AccountStatus::SUSPENDED->value) {
        throw new AppException(__('message.invalid_status_action'), 400);
      }

      switch ($validated['status']) {
        case AccountStatus::APPROVED->value: {

            $doctor->update(['status' => $validated['status']]);
            $data = [
              'homeUrl' => 'https://staging.alrahmah.om',
              'loginName' => $doctor->first_name,
            ];
            Mail::to($doctor->loginData->email)->send(new UserApprovedMail($data));
            Log::info("User approve email sent successfully to {$doctor->loginData->email}");

            return $this->success(__('message.doctor_approved'), $doctor, 200);
          }
        case AccountStatus::REJECTED->value: {
            $doctor->update(['status' => $validated['status'], 'reject_note' => $validated['note']]);
            $data = [
              'homeUrl' => 'https://staging.alrahmah.om',
              'loginName' => $doctor->first_name,
              'rejectReason' => 'dummy rejected reason'
            ];
            Mail::to($doctor->loginData->email)->send(new UserRejectedMail($data));
            Log::info("User reject email sent successfully to {$doctor->loginData->email}");

            return $this->success(__('message.doctor_rejected'), $doctor, 200);
          }
        case AccountStatus::SUSPENDED->value: {
            $doctor->update(['status' => $validated['status'], 'deleted_note' => $validated['note']]);
            $data = [
              'homeUrl' => 'https://staging.alrahmah.om',
              'loginName' => $doctor->first_name,
              'deleteReason' => 'dummy deleted reason'
            ];
            Mail::to($doctor->loginData->email)->send(new UserDeletededMail($data));
            Log::info("User deleted email sent successfully to {$doctor->loginData->email}");

            return $this->success(__('message.doctor_deleted'), $doctor, 200);
          }
        default: {
            throw new AppException(__('message.invalid_status'), 400);
          }
      }
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'));
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function createSlot(Request $request)
  {
    DB::beginTransaction();
    try {
      $validator = Validator::make($request->all(), [
        'doctorId' => 'nullable|exists:doctor,user_account_id',
        'startDate' => 'required|date',
        'endDate' => 'required|date|after_or_equal:startDate',
        'slots' => 'required|array',
        'slots.*.id' => 'required|exists:doctor_working_hour,id',
      ]);
      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();
      $user = JWTAuth::parseToken()->getPayload();
      $roleId = $user->get('roleId');
      $role = Role::tryFrom($roleId);

      $doctorId = ($role === Role::DOCTOR) ? $user->get('userAccountId') : $validated['doctorId'];
      if ($role === Role::ADMIN && empty($doctorId)) {
        return response()->json(['error' => 'Doctor ID is required for admin'], 400);
      }

      $startDate = Carbon::parse($validated['startDate']);
      $endDate = Carbon::parse($validated['endDate']);
      $createdWorkingDays = [];

      while ($startDate->lte($endDate)) {
        $currentDate = $startDate->format('Y-m-d');

        $workingDay = DoctorWorkingDay::where('doctor_id', $doctorId)
          ->where('working_day', $currentDate)
          ->first();

        if (!$workingDay) {
          $workingDay = DoctorWorkingDay::create([
            'doctor_id' => $doctorId,
            'working_day' => $currentDate,
            'status' => DoctorSlotStatus::OPEN->value
          ]);
        }
        $createdWorkingDays[] = [
          'id' => $workingDay->id,
          'doctor_id' => $doctorId,
          'working_day' => $workingDay->working_day,
          'status' => $workingDay->status,
        ];
        DoctorSlot::where('doctor_id', $doctorId)
          ->where('working_day_id', $workingDay->id)
          ->delete();

        foreach ($validated['slots'] as $slot) {
          DoctorSlot::create([
            'doctor_id' => $doctorId,
            'working_day_id' => $workingDay->id,
            'working_hour_id' => $slot['id'],
            'status' => DoctorSlotStatus::OPEN->value
          ]);
        }

        $startDate->addDay();
      }

      DB::commit();
      return $this->success(__('message.success'), $createdWorkingDays, 201);
    } catch (AppException $e) {
      DB::rollBack();
      return $e->render($request);
    } catch (Exception $e) {
      DB::rollBack();
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  public function deleteWorkingDay(Request $request, $id)
  {
    DB::beginTransaction();
    try {
      $user = Auth::user();
      $role = Role::tryFrom(JWTAuth::parseToken()->getPayload()->get('roleId'));

      $workingDay = DoctorWorkingDay::findOrFail($id);

      if ($role === Role::DOCTOR && $workingDay->doctor_id !== $user->user_account_id) {
        throw new AppException(__('message.unauthorized'), 403);
      }

      DoctorSlot::where('working_day_id', $workingDay->id)->delete();

      $workingDay->delete();

      DB::commit();
      return $this->success(__('message.success'), null, 200);
    } catch (ModelNotFoundException $e) {
      DB::rollBack();
      return $this->error(404, __('message.not_found'));
    } catch (AppException $e) {
      DB::rollBack();
      return $e->render($request);
    } catch (Exception $e) {
      DB::rollBack();
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }




  public function getworkingDays(Request $request)
  {
    try {
      $validator = Validator::make($request->query(), [
        'doctorId' => 'nullable|exists:doctor,user_account_id',
        'startDate' => 'nullable|date',
        'endDate'   => 'nullable|date|after_or_equal:start_date',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }
      $validated = $validator->validated();
      $doctorId = $validated['doctorId'] ?? null;
      $startDate = $validated['startDate'] ?? null;
      $endDate   = $validated['endDate'] ?? null;

      $workingDays = DoctorWorkingDay::where('doctor_id', $doctorId)
        ->when($startDate && $endDate, function ($q) use ($startDate, $endDate) {
          $q->whereBetween('working_day', [$startDate, $endDate]);
        })
        ->with([
          'slots' => function ($query) {
            $query->with([
              'workingDay:id,working_day',
              'workingHour:id,start_time,end_time'
            ]);
          }
        ])->orderBy('working_day', 'asc')
        ->get();

      return $this->success(__('message.success'), $workingDays, 200);
    } catch (AppException $e) {

      return $e->render($request);
    } catch (Exception $e) {
      return response()->json(['error' => __('message.server_error'), 'details' => $e->getMessage()], 500);
    }
  }
  public function submitReview(Request $request, string $bookingId)
  {
    try {
      $user = JWTAuth::parseToken()->getPayload();
      $userId = $user->get('userAccountId');

      $booking = DoctorBooking::findOrFail($bookingId);

      $validator = Validator::make($request->all(), [
        'rating' => 'required|integer|min:1|max:5',
        'review' => 'nullable|string',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }
      $validated = $validator->validated();
      $review = DoctorReview::updateOrCreate(
        ['booking_id' => $bookingId],
        [
          'doctor_id' => $booking->doctor_id,
          'patient_id' => $userId,
          'rating' => $validated['rating'],
          'review' => $validated['review'] ?? null,
        ]
      );

      return $this->success(__('message.success'), $review, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'), []);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return response()->json(['error' => __('message.server_error'), 'details' => $e->getMessage()], 500);
    }
  }
  public function likeReview($reviewId)
  {
    try {
      $user = JWTAuth::parseToken()->getPayload();
      $userId = $user->get('userAccountId');

      $like = ReviewLike::firstOrCreate([
        'review_id' => $reviewId,
        'patient_id' => $userId,
      ]);

      return $this->success(__('message.success'), null, 200);
    } catch (Exception $e) {
      return response()->json(['error' => __('message.server_error'), 'details' => $e->getMessage()], 500);
    }
  }
  public function getDoctorReviews(Request $request, $doctorId)
  {
    try {
      $validator = Validator::make($request->all(), [
        'page' => 'integer|min:1',
        'size' => 'integer|min:1|max:100',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();
      $page = (int) ($validated['page'] ?? 1);
      $size = (int) ($validated['size'] ?? 10);
      $offset = ($page - 1) * $size;

      $query = DoctorReview::with('patient:id,first_name,last_name')
        ->where('doctor_id', $doctorId);

      $total = $query->count();

      $reviews = $query->orderBy('created_at', 'desc')
        ->offset($offset)
        ->limit($size)
        ->get();

      return $this->success(__('message.success'), [
        'data' => $reviews,
        'pagination' => [
          'total' => $total,
          'current_page' => $page,
          'per_page' => $size,
          'last_page' => ceil($total / $size),
        ]
      ], 200);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
}
