<?php

namespace App\Http\Controllers;

use App\Exceptions\AppException;
use App\Http\Controllers\Controller;
use App\Models\DiagnosisFormField;
use App\Models\Doctor;
use App\Traits\ApiResponse;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Tymon\JWTAuth\Facades\JWTAuth;

class DiagnosisFormController extends Controller
{
  use ApiResponse;
  public function index(Request $request)
  {
    try {
      $user = JWTAuth::parseToken()->getPayload();
      $userId = $user->get('userAccountId');
      $doctor = Doctor::where('user_account_id', $userId)->first();

      if (!$doctor) {
        throw new AppException(__('message.unauthorized'), 401);
      }

      log::info('doctor', [$doctor, $userId]);
      $designationId = $doctor->designation_id;
      $diagnosisForm = DiagnosisFormField::query()->where('designation_id', $designationId)
        ->orderBy('order', 'asc')
        ->get();

      if ($diagnosisForm->isEmpty()) {
        $diagnosisForm = DiagnosisFormField::query()->whereNull('designation_id')
          ->orderBy('order', 'asc')
          ->get();
      }
      return $this->success(__(key: 'message.success'), [
        'diagnosisFormFields' => $diagnosisForm,
      ], 200);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function show(Request $request, string $designationId)
  {
    try {
      $validator = Validator::make(['designationId' => $designationId], [
        'designationId' => 'required|integer|exists:designation,id',
      ]);
      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }
      $validated = $validator->validated();
      $designationId = $validated['designationId'];
      $diagnosisForm = DiagnosisFormField::query()->where('designation_id', $designationId)
        ->orderBy('order', 'asc')
        ->get();


      return $this->success(__(key: 'message.success'), [
        'diagnosisFormFields' => $diagnosisForm,
      ], 200);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function store(Request $request)
  {
    try {
      $validator = Validator::make($request->all(), [
        'designation_id' => 'required|integer|exists:designation,id',
        'fields' => 'required|array',
        'fields.*.name' => 'required|string',
        'fields.*.label' => 'required|string',
        'fields.*.type' => 'required|string|in:text,textarea,select,checkbox,radio,number,date,progress_bar',
        'fields.*.options' => 'nullable|array',
        'fields.*.is_required' => 'required|boolean',
        'fields.*.group' => 'nullable|string',
        'fields.*.order' => 'required|integer',
        'fields.*.designation_id' => 'nullable|integer|exists:designation,id',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();
      $designationId = $validated['designation_id'] ?? null;
      $createdFields = [];

      foreach ($validated['fields'] as $field) {
        $createdFields[] = DiagnosisFormField::create([
          'name' => $field['name'],
          'label' => $field['label'],
          'type' => $field['type'],
          'options' => isset($field['options']) ? json_encode($field['options']) : null,
          'is_required' => $field['is_required'],
          'group' => $field['group'] ?? null,
          'order' => $field['order'],
          'designation_id' => $designationId,
        ]);
      }

      return $this->success(__('message.created_successfully'), $createdFields, 201);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function update(Request $request, string $designationId)
  {
    try {
      $validator = Validator::make(array_merge($request->all(), ['designation_id' => $designationId]), [
        'designation_id' => 'required|integer|exists:designation,id',
        'fields' => 'required|array',
        'fields.*.name' => 'required|string',
        'fields.*.label' => 'required|string',
        'fields.*.type' => 'required|string|in:text,textarea,select,checkbox,radio,number,date,progress_bar',
        'fields.*.options' => 'nullable|array',
        'fields.*.is_required' => 'required|boolean',
        'fields.*.group' => 'nullable|string',
        'fields.*.order' => 'required|integer',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();

      // Delete existing fields for the designation
      DiagnosisFormField::where('designation_id', $designationId)->delete();

      $newFields = array_map(function ($field) use ($designationId) {
        return [
          'name' => $field['name'],
          'label' => $field['label'],
          'type' => $field['type'],
          'options' => isset($field['options']) ? json_encode($field['options']) : null,
          'is_required' => $field['is_required'],
          'group' => $field['group'] ?? null,
          'order' => $field['order'],
          'designation_id' => $designationId,
          'created_at' => now(),
          'updated_at' => now(),
        ];
      }, $validated['fields']);

      DiagnosisFormField::insert($newFields);

      return $this->success(__('message.updated_successfully'), [], 200);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
}
