<?php

namespace App\Http\Controllers;

use App\Exceptions\AppException;
use App\Http\Controllers\Controller;
use App\Models\Designation;
use App\Traits\ApiResponse;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Log;
use Validator;

class DesignationController extends Controller
{
  use ApiResponse;

  public function index(Request $request)
  {
    try {
      $validator = Validator::make($request->all(), [
        'page' => 'integer|min:1',
        'size' => 'integer|min:1|max:100',
        'search' => 'string|nullable'
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }
      $validated = $validator->validated();
      $page = (int) ($validated['page'] ?? 1);
      $size = (int) ($validated['size'] ?? 10);
      $search = $validated['search'] ?? null;
      $offset = ($page - 1) * $size;

      $designationsQuery = Designation::query();

      if (!empty($search)) {
        $designationsQuery->where('name', 'like', '%' . $search . '%');
      }

      $total = $designationsQuery->count();

      $designations = $designationsQuery
        ->orderBy('created_at', 'desc')
        ->offset($offset)
        ->limit($size)
        ->get();

      return $this->success(__(key: 'message.success'), [
        'designations' => $designations,
        'pagination' => [
          'total' => $total,
          'current_page' => $page,
          'per_page' => $size,
          'last_page' => ceil($total / $size)
        ]
      ], 200);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  public function show($id)
  {
    try {
      $designation = Designation::findOrFail($id);
      return $this->success(__('message.success'), $designation, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.designation_not_found'));
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  public function store(Request $request)
  {
    try {
      $validator = Validator::make($request->all(), [
        'name' => 'required|string|max:255',
        'description' => 'nullable|string',
        'name_ar' => 'nullable|string|max:255',
        'description_ar' => 'nullable|string',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();
      $designation = Designation::create($validated);

      return $this->success(__('message.designation_created'), $designation, 201);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }


  public function update(Request $request, $id)
  {
    try {
      $designation = Designation::findOrFail($id);

      $validator = Validator::make($request->all(), [
        'name' => 'required|string|max:255',
        'description' => 'nullable|string',
        'name_ar' => 'nullable|string|max:255',
        'description_ar' => 'nullable|string'
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();
      $designation->update($validated);

      return $this->success(__('message.designation_updated'), $designation, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.designation_not_found'));
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  public function destroy($id)
  {
    try {
      $designation = Designation::findOrFail($id);
      $designation->delete();
      return $this->success(__('message.designation_deleted'), null, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.designation_not_found'));
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }


  public function publicIndex()
  {
    try {
      $designations = Designation::select('id', 'name', 'name_ar')->get();

      return $this->success(__('message.success'), $designations, 200);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
}
