<?php

namespace App\Http\Controllers;

use App\Enums\BookingStatus;
use App\Enums\DoctorSlotStatus;
use App\Enums\Role;
use App\Exceptions\AppException;
use App\Http\Controllers\Controller;
use App\Http\Resources\BookingListResource;
use App\Http\Resources\BookingResource;
use App\Http\Resources\DoctorReportResource;
use App\Models\Consultation;
use App\Models\DiagnosisFormSubmission;
use App\Models\DiagnosisFormValue;
use App\Models\DoctorBooking;
use App\Models\DoctorSlot;
use App\Models\PatientPrescription;
use App\Models\VideoConsultation;
use App\Services\BMeetService;
use App\Traits\ApiResponse;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Tymon\JWTAuth\Facades\JWTAuth;

use function Illuminate\Log\log;

class BookingController extends Controller
{
  use ApiResponse;
  protected $bmeetService;
  public function __construct(BMeetService $bmeetService)
  {
    $this->bmeetService = $bmeetService;
  }
  /**
   * Display a listing of the resource.
   */
  public function index(Request $request)
  {
    try {
      $user = JWTAuth::parseToken()->getPayload();
      $role = $user->get('roleId');
      $userId = $user->get('userAccountId');

      // Validate request parameters
      $validator = Validator::make($request->all(), [
        'page'      => 'integer|min:1',
        'size'      => 'integer|min:1|max:100',
        'status'    => ['nullable', Rule::enum(BookingStatus::class)],
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      // Extract validated parameters
      $validated = $validator->validated();
      $page      = (int) ($validated['page'] ?? 1);
      $size      = (int) ($validated['size'] ?? 10);
      $status    = $validated['status'] ?? null;
      $offset    = ($page - 1) * $size;

      $query = DoctorBooking::with([
        'patient:id,first_name,last_name,country_code,mobile_number,avatar_url',
        'doctor' => function ($q) {
          $q->select('user_account_id', 'license_no', 'qualification', 'experience', 'department_id')
            ->withAvg('reviews as average_rating', 'rating')
            ->withCount('reviews')
            ->with([
              'userAccount:id,first_name,last_name,country_code,mobile_number,avatar_url',
              'userLoginData:email',
              'department:id,name,name_ar',
            ]);
        },
        'slot.workingDay',
        'slot.workingHour'
      ])
        ->select('id', 'doctor_id', 'patient_id', 'slot_id', 'visit_reason', 'symtoms', 'status');

      // Apply filters
      if ($role == Role::USER->value) {
        $query->where('patient_id', $userId);
      }
      if ($role == Role::DOCTOR->value) {
        $query->where('doctor_id', $userId);
      }
      if ($status) {
        $query->where('status', $status);
      }

      // Get total count for pagination
      $total = $query->count();

      // Get paginated results
      $bookings = $query
        ->orderBy('booking_date', 'desc')
        ->offset($offset)
        ->limit($size)
        ->get();
      Log::info('Bookings Data:', ['bookings' => $bookings->toArray()]);
      return $this->success(__('message.success'), [
        'bookings' => $bookings->isEmpty() ? [] : BookingListResource::collection($bookings),
        'pagination' => [
          'total'         => $total,
          'current_page'  => $page,
          'per_page'      => $size,
          'last_page'     => ceil($total / $size),
        ]
      ], 200);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  /**
   * Store a newly created resource in storage.
   */
  public function store(Request $request, $slotId)
  {
    DB::beginTransaction();
    try {
      $user = JWTAuth::parseToken()->getPayload();
      $patientId = $user->get('userAccountId');

      $validator = Validator::make($request->all(), [
        'prescriptions.*' => 'nullable|file|mimes:jpg,jpeg,png,pdf,docx|max:20480',
        'visit_reason' => 'nullable|string|max:255',
        'symptoms' => 'nullable|string',
      ]);
      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }
      $validated = $validator->validated();
      $slot = DoctorSlot::where('id', $slotId)->where('status', DoctorSlotStatus::OPEN->value)->first();
      if (!$slot) {
        throw new AppException(__('message.not_found'), 404);
      }

      $slot->update(['status' => DoctorSlotStatus::BOOKED->value]);

      $appointment = DoctorBooking::create([
        'patient_id' => $patientId,
        'doctor_id' => $slot->doctor_id,
        'slot_id' => $slot->id,
        'booking_date' => Carbon::now()->utc(),
        'visit_reason' => $validated['visitReason'] ?? null,
        'symptoms' => $validated['symptoms'] ?? null,
        'status' => BookingStatus::SCHEDULED->value
      ]);

      if ($request->hasFile('prescriptions')) {
        foreach ($request->file('prescriptions') as $file) {
          $fileName = time() . '-' . $file->getClientOriginalName();
          $file->storeAs('prescriptions', $fileName);
          $path = '/storage/prescriptions/' . $fileName;

          PatientPrescription::create([
            'booking_id' => $appointment->id,
            'file_url' => $path,
            'original_name' => $file->getClientOriginalName(),
            'mime_type' => $file->getMimeType(),
            'size' => $file->getSize(),
          ]);
        }
      }
      $booking = DoctorBooking::with(['doctor.userAccount', 'patient', 'slot.workingHour', 'slot.workingDay',])->find($appointment->id);
      Log::info("test", [$booking]);
      $this->scheduleVideoConsultation($booking);
      DB::commit();

      return $this->success(__('message.booking_confirmed'), ['appointment' => $appointment], 200);
    } catch (ModelNotFoundException $e) {
      DB::rollBack();
      return $this->error(404, __('message.not_found'));
    } catch (AppException $e) {
      DB::rollBack();
      return $e->render($request);
    } catch (Exception $e) {
      DB::rollBack();
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  /**
   * Schedule a video consultation for a booking
   */
  protected function scheduleVideoConsultation($booking)
  {
    // Create meeting in BMeet
    $meetingData = $this->bmeetService->createMeeting($booking);

    // Save meeting details
    VideoConsultation::create([
      'booking_id' => $booking->id,
      'meeting_id' => $meetingData['mid'],
      'm_id' => $meetingData['m_id'],
      'conference_url' => $meetingData['conference_url'] ?? null,
      'scheduled_start' => Carbon::parse($booking->slot->workingDay->working_day . ' ' . $booking->slot->workingHour->start_time),
      'scheduled_end' => Carbon::parse($booking->slot->workingDay->working_day . ' ' . $booking->slot->workingHour->end_time),
    ]);
  }

  /**
   * Display the specified resource.
   */
  public function show(string $id)
  {
    try {
      $user = JWTAuth::parseToken()->getPayload();
      $role = $user->get('roleId');
      $userId = $user->get('userAccountId');

      $query = DoctorBooking::with([
        'patient:id,first_name,last_name,country_code,mobile_number,avatar_url',
        'doctor' => function ($q) {
          $q->select('user_account_id', 'license_no', 'qualification', 'experience', 'department_id')
            ->withAvg('reviews as average_rating', 'rating')
            ->withCount('reviews')
            ->with([
              'userAccount:id,first_name,last_name,country_code,mobile_number,avatar_url',
              'userLoginData:email',
              'department:id,name,name_ar',
            ]);
        },
        'slot.workingDay',
        'slot.workingHour'
      ])
        ->select('id', 'doctor_id', 'patient_id', 'slot_id', 'visit_reason', 'symtoms', 'status')->where('id', $id);

      if ($role == Role::USER->value) {
        $query->where('patient_id', $userId);
      }
      if ($role == Role::DOCTOR->value) {
        $query->where('doctor_id', $userId);
      }
      $appointment = $query
        ->get()->first();

      if (!$appointment) {
        return $this->error(404, __('message.not_found'), []);
      }
      return $this->success(__('message.success'), ['booking' => new BookingResource($appointment)], 200);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  /**
   * Update the specified resource in storage.
   */
  public function update(Request $request, string $id)
  {
    try {
      $appointment = DoctorBooking::findOrFail($id);

      $validator = Validator::make($request->all(), [
        'rescheduleReason' => 'nullable|string',
        'slotId' => 'required|integer|exists:doctor_slot,id',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();
      $appointment->update([
        'reschedule_reason' => $validated['rescheduleReason'],
        'slot_id' => $validated['slotId']
      ]);

      return $this->success(__('message.booking_rescheduled'), $appointment, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'), []);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function action(Request $request, string $id)
  {
    try {
      $validator = Validator::make($request->all(), [
        'status' => 'required|integer|in:3',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();

      $appointment = DoctorBooking::findOrFail($id);
      $appointment->update([
        'status' => $validated['status']
      ]);
      return $this->success(__('message.booking_canceled'), $appointment, 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'), []);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }

  /**
   * Remove the specified resource from storage.
   */
  public function destroy(string $id)
  {
    //
  }
  public function doctorReports(Request $request, string $bookingId)
  {
    DB::beginTransaction();
    try {
      $validator = Validator::make(array_merge($request->all(), ['booking_id' => $bookingId]), [
        'booking_id' => 'required|integer|exists:patient_booking,id',
        'fields' => 'required|array',
        'fields.*.field_id' => 'required|integer|exists:diagnosis_form_field,id',
        'fields.*.value' => 'required',
        'consultation_date' => 'required|date',
        'consultation_time' => 'required',
        'additional_observations' => 'nullable|string',
        'behavioural_observations' => 'nullable|string',
        'emotional_observations' => 'nullable|string',
        'cognitive_observations' => 'nullable|string',
        'examination' => 'nullable|string',
        'plan' => 'nullable|string',
        'next_appointment_date' => 'nullable|date',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();

      // Create diagnosis form submission
      $submission = DiagnosisFormSubmission::create([
        'booking_id' => $bookingId,
        'submitted_at' => now(),
      ]);

      // Save form values
      foreach ($validated['fields'] as $field) {
        DiagnosisFormValue::create([
          'submission_id' => $submission->id,
          'field_id' => $field['field_id'],
          'value' => json_encode($field['value']),
        ]);
      }
      // Fetch values with field info
      $diagnosisValues = DiagnosisFormValue::with('field')
        ->where('submission_id', $submission->id)
        ->get()
        ->map(function ($item) {
          return [
            'field_id' => $item->field_id,
            'field_name' => $item->field->name ?? null,
            'value' => json_decode($item->value),
          ];
        });

      // Create consultation record
      $consultation = Consultation::create([
        'booking_id' => $bookingId,
        'consultation_date' => $validated['consultation_date'],
        'consultation_time' => $validated['consultation_time'],
        'additional_observations' => $validated['additional_observations'] ?? null,
        'behavioural_observations' => $validated['behavioural_observations'] ?? null,
        'emotional_observations' => $validated['emotional_observations'] ?? null,
        'cognitive_observations' => $validated['cognitive_observations'] ?? null,
        'examination' => $validated['examination'] ?? null,
        'plan' => $validated['plan'] ?? null,
        'next_appointment_date' => $validated['next_appointment_date'] ?? null,
      ]);
      $booking = DoctorBooking::findOrFail($bookingId);
      $booking->update([
        'status' => BookingStatus::COMPLETED->value
      ]);
      DB::commit();

      return $this->success(__('message.consultation_visited'), [
        'consultation' => $consultation,
        'diagnosis_values' => $diagnosisValues,
      ], 200);
    } catch (Exception $e) {
      DB::rollBack();
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function getPrevDoctorReports(string $bookingId)
  {
    try {
      $user = JWTAuth::parseToken()->getPayload();
      $role = $user->get('roleId');
      $userId = $user->get('userAccountId');

      $currentBooking = DoctorBooking::findOrFail($bookingId);

      // Build query for previous bookings (excluding current one)
      $query = DoctorBooking::where('doctor_id', $currentBooking->doctor_id)
        ->where('patient_id', $currentBooking->patient_id)
        ->where('id', '<>', $bookingId);

      // Add role-based filters
      if ($role == Role::USER->value) {
        $query->where('patient_id', $userId);
      }

      if ($role == Role::DOCTOR->value) {
        $query->where('doctor_id', $userId);
      }

      // Now fetch the IDs
      $previousBookings = $query->pluck('id');

      $doctorReport = Consultation::with([
        'diagnosisSubmission.diagnosisValues.field',
      ])->whereIn('booking_id', $previousBookings)->get();


      return $this->success(__('message.success'), [
        'doctorReports' => $previousBookings->isEmpty() ? [] : DoctorReportResource::collection($doctorReport)
      ]);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
}
