<?php

namespace App\Http\Controllers;

use App\Enums\AccountStatus;
use App\Enums\Role;
use App\Enums\SubscriptionStatus;
use App\Exceptions\AppException;
use App\Http\Controllers\Controller;
use App\Mail\PasswordResetOtpMail;
use App\Models\Doctor;
use App\Models\DoctorCertificate;
use App\Models\DoctorIdsDocument;
use App\Models\Plan;
use App\Models\Subscription;
use App\Models\UserAccount;
use App\Models\UserLoginData;
use App\Traits\ApiResponse;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use Exception;
use Hash;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Mail;
use Str;
use Tymon\JWTAuth\Exceptions\JWTException;
use Tymon\JWTAuth\Exceptions\TokenInvalidException;
use Tymon\JWTAuth\Facades\JWTAuth;
use Validator;

class AuthController extends Controller
{
  use ApiResponse;
  protected string $malpracticeFilePath;
  public function __construct()
  {
    $this->malpracticeFilePath = 'doctor/medical_malpractice';
  }
  //
  public function register(Request $request)
  {
    DB::beginTransaction();
    try {
      $isDoctor  = $request->roleId == Role::DOCTOR->value;

      $commonRules = [
        'avatar' => 'required|file|mimes:jpg,jpeg,png|max:20480',
        'firstName' => 'required|string|max:255',
        'lastName' => 'required|string|max:255',
        'nationalId' => 'required|string|unique:user_account,national_id',
        'nationalIdExpiryAt' => 'required|date',
        'countryCode' => 'required|string',
        'mobile' => 'required|string|unique:user_account,mobile_number',
        'emergencyCountryCode' => 'required|string',
        'emergencyMobile' => 'required|string',
        'email' => 'required|email|unique:user_login_data,email',
        'dateOfBirth' => 'required|date',
        'gender' => 'required|string',
        'roleId' => 'required|integer|in:2,3',
        'password' => 'required|string|min:6'
      ];
      if ($isDoctor) {

        $commonRules += [
          'licenseNo' => 'required|string|max:255|regex:/^[A-Za-z0-9\-]+$/|unique:doctor,license_no',
          'specializationId' => 'required|integer',
          'experience' => 'required|numeric|min:1|max:99',
          'qualification' => 'required|string|max:255',
          'medicalSchool' => 'required|string|max:255',
          'departmentId' => 'required|integer',
          'designationId' => 'required|integer',
          'bio' => 'required|string|max:1055',
          'areasOfExpertise' => 'required|string|max:1055',
          'medicalMalpractice' => 'required|file|mimes:jpg,jpeg,png,pdf,doc,docx|max:20480',
          'certifications' => 'required|array',
          'certifications.*' => 'file|mimes:pdf,jpg,jpeg,png|max:20480',
          'identificationDocuments' => 'required|array',
          'identificationDocuments.*' => 'file|mimes:pdf,jpg,jpeg,png|max:20480',
          'specializationServiceId' => 'required|integer',
          'avrgConsultationTime' => 'required|integer|min:15|max:300',
          'avrgConsultationFee' => 'required|numeric|min:0.01|max:10000'
        ];
      }

      $validator = Validator::make($request->all(), $commonRules);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();
      $avatar = $validated['avatar'];

      $fileName = time() . '-' . $avatar->getClientOriginalName();
      $avatar->storeAs('profile', $fileName, 'public');
      $path = '/storage/profile/' . $fileName;

      $userAccount = UserAccount::create([
        'avatar_url' => $path,
        'first_name' => $validated['firstName'],
        'last_name' => $validated['lastName'],
        'national_id' => $validated['nationalId'],
        'national_id_exp_at' => $validated['nationalIdExpiryAt'],
        'country_code' => $validated['countryCode'],
        'mobile_number' => $validated['mobile'],
        'emergency_country_code' => $validated['emergencyCountryCode'],
        'emergency_mobile_number' => $validated['emergencyMobile'],
        'date_of_birth' => $validated['dateOfBirth'],
        'gender' => $validated['gender'],
        'role_id' => $validated['roleId'],
        'status' => $isDoctor ? AccountStatus::PENDING->value : AccountStatus::APPROVED->value
      ]);

      $baseLoginName = strtolower($validated['firstName'] . '.' . $validated['lastName']);
      $loginName = $baseLoginName;
      $count = 1;

      while (UserLoginData::where('login_name', $loginName)->exists()) {
        $loginName = $baseLoginName . $count;
        $count++;
      }

      UserLoginData::create([
        'user_account_id' => $userAccount->id,
        'login_name' => $loginName,
        'email' => $validated['email'],
        'password_hash' => Hash::make($validated['password']),
      ]);

      if ($isDoctor) {

        $plan = Plan::where('is_trial', 1)->first();
        if (!$plan) {
          throw new AppException('Trial plan not found.', 404);
        }
        Subscription::create([
          'doctor_id' => $userAccount->id,
          'plan_id' => $plan->id,
          'start_date' => Carbon::now('UTC'),
          'end_date' => Carbon::now('UTC')->addMonths($plan->duration_months),
          'status' => SubscriptionStatus::ACTIVE->value
        ]);

        $malpractice = $validated['medicalMalpractice'];
        $malpracticeFileName = time() . '-' . $malpractice->getClientOriginalName();
        $malpractice->storeAs($this->malpracticeFilePath, $malpracticeFileName);
        $doctor = Doctor::create([
          'user_account_id' => $userAccount->id,
          'license_no' => $validated['licenseNo'],
          'specialization_id' => $validated['specializationId'],
          'experience' => $validated['experience'],
          'qualification' => $validated['qualification'],
          'medical_school' => $validated['medicalSchool'],
          'department_id' => $validated['departmentId'],
          'designation_id' => $validated['designationId'],
          'bio' => $validated['bio'],
          'areas_of_expertise' => $validated['areasOfExpertise'],
          'medical_malpractice' => $malpracticeFileName,
          'specialization_service_id' => $validated['specializationServiceId'],
          'avrg_consultation_time' => $validated['avrgConsultationTime'],
          'avrg_consultation_fee' => $validated['avrgConsultationFee']
        ]);

        // Handle certifications
        foreach ($request->file('certifications') as $file) {
          $path = $file->store('doctor/certifications');
          DoctorCertificate::create([
            'doctor_id' => $doctor->user_account_id,
            'original_name' => $file->getClientOriginalName(),
            'file_url' => $path,
            'mime_type' => $file->getMimeType(),
            'size' => $file->getSize(),
          ]);
        }

        // Handle identification documents
        foreach ($request->file('identificationDocuments') as $file) {
          $path = $file->store('doctor/identifications');
          DoctorIdsDocument::create([
            'doctor_id' => $doctor->user_account_id,
            'original_name' => $file->getClientOriginalName(),
            'file_url' => $path,
            'mime_type' => $file->getMimeType(),
            'size' => $file->getSize(),
          ]);
        }
      }

      DB::commit();

      return $this->success(__('message.registration_success'), ['user' => $userAccount], 200);
    } catch (AppException $e) {
      DB::rollback();
      return $e->render($request);
    } catch (Exception $e) {
      DB::rollback();
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function login(Request $request)
  {
    try {
      $validator = Validator::make($request->all(), [
        'email' => 'required|email',
        'password' => 'required|string'
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $credentials = $validator->validated();

      if (!$token = JWTAuth::attempt($credentials)) {
        throw new AppException(__('message.invalid_credential'), 404);
      }

      $userLogin = Auth::user();

      $userAccount = UserAccount::where('id', $userLogin->user_account_id)
        ->where('status', AccountStatus::APPROVED->value)
        ->first();

      if (!$userAccount) {
        throw new AppException(__('message.invalid_credential'), 404);
      }

      $customClaims = [
        'email' => $userLogin->email,
        'firstName' => $userAccount->first_name,
        'lastName' => $userAccount->last_name,
        'roleId' => $userAccount->role_id,
        'userAccountId' => $userAccount->id
      ];

      $token = JWTAuth::claims($customClaims)->fromUser($userLogin);
      $userData = [
        'firstName' => $userAccount->first_name,
        'lastName' => $userAccount->last_name,
        'email' => $userLogin->email,
        'roleId' => $userAccount->role_id,
      ];
      switch ($userAccount->role_id) {
        case Role::ADMIN->value: {
            return $this->success(__('message.login_success'), [
              'token' => $token,
              'user' => $userData
            ], 200);
          }
        case Role::DOCTOR->value: {
            $subcription =  Subscription::where('doctor_id', $userAccount->id)->where('status', SubscriptionStatus::ACTIVE->value)->first();
            if (!$subcription) {

              return $this->success(__('message.subscrription_upgrade'), ['user' => $userData], 200);
            }
            return $this->success(__('message.login_success'), [
              'token' => $token,
              'user' => $userData,
              ' subcription' => ['start_date' => $subcription->start_date, 'end_date' => $subcription->end_date]
            ], 200);
          }
        case Role::USER->value: {
            return $this->success(__('message.login_success'), [
              'token' => $token,
              'user' => $userData
            ], 200);
          }
        default:
          throw new AppException(__('message.invalid_role'), 403);
      }
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function logout()
  {
    try {
      JWTAuth::invalidate(true);

      return $this->success(__('message.logout_success'), null, 200);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function passwordResetRequest(Request $request)
  {
    try {
      $validator = Validator::make($request->all(), [
        'email' => 'required|email',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }
      $validated = $validator->validate();
      $userLoginData = UserLoginData::where('email', $validated['email'])->first();

      if (!$userLoginData) {
        throw new AppException(__('message.pwdResetReqRes'), 401);
      }

      $resetOtp = mt_rand(1000, 9999);
      $resetExpAt = Carbon::now('UTC')->addMinutes((int) config('auth.password_reset_otp_exp'));

      $userLoginData->update([
        'reset_otp' => $resetOtp,
        'reset_otp_exp_at' => $resetExpAt,
      ]);

      $appConfig = config('app');
      $data = [
        'homeUrl' => $appConfig['url'],
        'loginName' => $userLoginData->login_name,
        'otp' => $resetOtp,
      ];
      Mail::to($request->email)->send(new PasswordResetOtpMail($data));
      Log::info("Password reset OTP email sent successfully to {$request->email}");

      return $this->success(__('message.pwdResetReqRes'), null, 200);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      Log::error("Failed to send password reset OTP email to {$request->email}. Error: " . $e->getMessage());
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function passwordResetOtpRequest(Request $request)
  {
    try {
      $validator = Validator::make($request->all(), [
        'email' => 'required|email',
        'otp' => 'required|string|size:4',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validate();
      $userLoginData = UserLoginData::where('email', $validated['email'])->first();

      if (!$userLoginData) {
        throw new AppException(__('message.pwdResetReqRes'), 401);
      }

      if ($userLoginData->reset_otp !== $validated['otp']) {
        throw new AppException(__('message.invalid_otp'), 401);
      }

      if (!$userLoginData->reset_otp_exp_at || Carbon::parse($userLoginData->reset_otp_exp_at)->isPast()) {
        throw new AppException(__('message.otp_expired'), 401);
      }

      $resetToken = Str::random(20);
      $resetExpAt = Carbon::now('UTC')->addMinutes((int) config('auth.password_reset_token_exp'));

      $userLoginData->update([
        'reset_otp' => null,
        'reset_otp_exp_at' => null,
        'reset_token' => $resetToken,
        'reset_token_exp_at' => $resetExpAt
      ]);


      return $this->success(__('message.otp_verified'), ['token' => $userLoginData->getAttribute('reset_token')], 200);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function passwordResetConfirm(Request $request)
  {
    try {
      $validator = Validator::make($request->all(), [
        'password' => 'required|string|min:6',
        'token' => 'required|string',
      ]);

      if ($validator->fails()) {
        throw new AppException(__('message.validation_error'), 400, $validator->errors());
      }

      $validated = $validator->validated();
      $actionAt = Carbon::now('UTC');

      $userLoginData = UserLoginData::where('reset_token', $validated['token'])
        ->first();

      if (!$userLoginData) {
        throw new AppException(__('message.invalid_reset_token'), 400);
      }

      if (!$userLoginData->reset_token_exp_at || Carbon::parse($userLoginData->reset_token_exp_at)->isPast()) {
        throw new AppException(__('message.reset_token_expired'), 400);
      }

      $passwordHash = Hash::make($validated['password']);

      $userLoginData->update([
        'password_hash' => $passwordHash,
        'reset_token' => null,
        'reset_otp_exp_at' => null,
        'updated_at' => $actionAt,
      ]);

      return $this->success(__('message.password_reset_success'), null, 200);
    } catch (AppException $e) {
      return $e->render($request);
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
  public function refresh()
  {
    try {
      $newToken = Auth::refresh();
      $userLogin = Auth::user();
      return $this->success(__('message.password_reset_success'),  [
        'status' => 'success',
        'user' => [
          'user_account_id' => $userLogin->user_account_id,
          'email' => $userLogin->email,
        ],
        'authorisation' => [
          'token' => $newToken,
          'type' => 'bearer',
        ]
      ], 200);
    } catch (TokenInvalidException $e) {
      return $this->error(401, __('message.invalid_token'));
    } catch (JWTException $e) {
      return $this->error(401, __('message.token_not_provided'));
    }
  }
  public function profile()
  {
    try {
      $userToken = JWTAuth::parseToken()->getPayload();
      $userAccountId = $userToken->get('userAccountId');

      $user = UserAccount::from('user_account as ua')
        ->select(
          'ua.id',
          'ua.avatar_url',
          'ua.first_name',
          'ua.last_name',
          'ua.country_code',
          'ua.mobile_number',
          'ua.emergency_country_code',
          'ua.emergency_mobile_number',
          'ul.email',
          'ua.created_at',
          'ua.updated_at',
        )
        ->leftJoin('user_login_data as ul', 'ua.id', '=', 'ul.user_account_id')
        ->where('ua.id', $userAccountId)
        ->firstOrFail();

      return $this->success(__('message.success'), ['user' => $user], 200);
    } catch (ModelNotFoundException $e) {
      return $this->error(404, __('message.not_found'));
    } catch (Exception $e) {
      return $this->error(500, __('message.server_error'), [$e->getMessage()]);
    }
  }
}
