<?php

namespace App\Exceptions;

use Illuminate\Foundation\Exceptions\Handler as ExceptionHandler;
use Throwable;
use Illuminate\Http\Request;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class Handler extends ExceptionHandler
{
  public function render($request, Throwable $exception)
  {
    $this->renderable(function (Throwable $e, $request) {
      if ($request->is('api/*') || $request->wantsJson()) {
        if ($e instanceof ValidationException) {
          return response()->json([
            'message' => 'Validation failed',
            'statusCode' => 422,
            'errors' => $e->validator->errors()
          ], 422);
        }

        if ($e instanceof AuthenticationException) {
          return response()->json([
            'message' => 'Unauthenticated',
            'statusCode' => 401,
          ], 401);
        }

        if ($e instanceof ModelNotFoundException) {
          return response()->json([
            'message' => 'Resource not found',
            'statusCode' => 404,
          ], 404);
        }

        if ($e instanceof NotFoundHttpException) {
          return response()->json([
            'message' => 'Route not found',
            'statusCode' => 404,
          ], 404);
        }

        if ($e instanceof MethodNotAllowedHttpException) {
          return response()->json([
            'message' => 'Method not allowed',
            'statusCode' => 405,
          ], 405);
        }

        // Let AppException handle itself using its render method
        if ($e instanceof AppException) {
          return $e->render($request);
        }

        // For any other exceptions
        return response()->json([
          'message' => 'Server Error',
          'statusCode' => 500,
          'errors' => config('app.debug') ? [
            'error' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine()
          ] : []
        ], 500);
      }

      return null; // Let Laravel handle non-API requests
    });
  }
}
